package com.zero.utils.jwt;


import cn.hutool.core.util.StrUtil;
import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTCreator;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.zero.utils.CustomJwtException;
import com.zero.utils.PublicResult;
import com.zero.utils.config.RedisUtil;
import com.zero.utils.config.Utils;
import com.zero.utils.enums.JwtEnums;
import com.zero.utils.enums.LoginRedisEnums;
import com.zero.utils.enums.ResultEnums;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.DigestUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * jwt工具类
 *
 * @author zero
 */
@Log4j2
public class JwtUtil {
    /**
     * 获取token
     *
     * @return
     */
    public static String getToken() {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (!Objects.isNull(requestAttributes)) {
            HttpServletRequest request = ((ServletRequestAttributes) requestAttributes).getRequest();
            // 拿到当前Header中Authorization的AccessToken(Shiro中getAuthzHeader方法已经实现)
            String token = request.getHeader(JwtEnums.ACCESS_TOKEN_NAME.getValue());
            if (null == token) {
                token = request.getParameter(JwtEnums.ACCESS_TOKEN_NAME.getValue());
            }
            return token;
        }
        return null;
    }

    /**
     * 校验token是否正确
     * <p>
     * 这里不抛异常只返回是否正确
     *
     * @param token Token
     * @return boolean 是否正确
     */
    static boolean verify(String token) {
        try {
            // 帐号加JWT私钥解密
            String secret = getClaim(token, JwtEnums.MAME.getValue()) + DigestUtils.md5DigestAsHex(JwtEnums.SECRET_KEY.getValue().getBytes());
            Algorithm algorithm = Algorithm.HMAC256(secret);
            JWTVerifier verifier = JWT.require(algorithm)
                    .build();
            verifier.verify(token);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 获得Token中的信息无需secret解密也能获得
     *
     * @param token
     * @param claim
     * @return java.lang.String
     */
    public static String getClaim(String token, String claim) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            // 只能输出String类型，如果是其他类型返回null
            return jwt.getClaim(claim).asString();
        } catch (Exception e) {
            //这里抛出去不会被全局异常捕获，因为在JwtFilter。isAccessAllowed中处理了
            throw new CustomJwtException(PublicResult.resultEnums(ResultEnums.TOKEN_ANALYSIS_0, e.getMessage()));
        }
    }

    /**
     * 获得Token中的信息无需secret解密也能获得
     *
     * @param claim
     * @return java.lang.String
     */
    public static String getClaim(String claim) {
        String token = getToken();
        return StrUtil.isBlank(token) ? null : getClaim(getToken(), claim);
    }

    /**
     * 构建token信息
     *
     * @param username
     * @return
     */
    public static String buildTokenInfo(String username) {
        String nowStr = String.valueOf(System.currentTimeMillis());
        //缓存  刷新token
        RedisUtil.set(String.format(LoginRedisEnums.REFRESH_ACCESS_TOKEN_PREFIX.getValue(), username), nowStr);
        Map<String, String> map = new HashMap<>(3);
        String realIp = Utils.getRealIp();
        map.put(JwtEnums.MAME.getValue(), username);
        map.put(JwtEnums.CURRENT_TIME_MILLIS.getValue(), nowStr);
        map.put(JwtEnums.IP.getValue(), realIp);
        return sign(map);
    }

    /**
     * 生成签名
     *
     * @param map token信息
     * @return java.lang.String 返回加密的Token
     */
    private static String sign(Map<String, String> map) {
        try {
            // 帐号加JWT私钥加密
            String secret = map.get(JwtEnums.MAME.getValue()) + DigestUtils.md5DigestAsHex(JwtEnums.SECRET_KEY.getValue().getBytes());
            // 此处过期时间是以毫秒为单位，所以乘以1000
            Date date = new Date(System.currentTimeMillis() + JwtEnums.JWT_EXPIRE_TIME.getIntKey() * 1000);
            Algorithm algorithm = Algorithm.HMAC256(secret);
            // token信息处理
            JWTCreator.Builder builder = JWT.create();
            for (Map.Entry<String, String> entry : map.entrySet()) {
                builder.withClaim(entry.getKey(), entry.getValue());
            }
            return builder.withExpiresAt(date).sign(algorithm);
        } catch (Exception e) {
            //这里抛出去不会被全局异常捕获，因为在JwtFilter。isAccessAllowed中处理了
            throw new CustomJwtException(PublicResult.resultEnums(ResultEnums.TOKEN_ENCRYPTION_0, e.getMessage()));
        }
    }
}
