package com.zero.utils.config;

import cn.hutool.core.util.StrUtil;
import eu.bitwalker.useragentutils.Browser;
import eu.bitwalker.useragentutils.OperatingSystem;
import eu.bitwalker.useragentutils.UserAgent;
import lombok.extern.log4j.Log4j2;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Field;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.*;


/**
 * 其他工具类
 *
 * @author zero
 */
@Log4j2
public class Utils {
    private static final List<String> IP_ADDRESS = Arrays.asList("0.0.0.0", "0:0:0:0:0:0:0:1", "localhost", "127.0.0.1");
    private static final String UNKNOWN = "unknown";

    /**
     * 获取真实IP
     *
     * @return 真实IP
     */
    public static String getRealIp() {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        String ip;
        if (!Objects.isNull(requestAttributes)) {
            HttpServletRequest httpServletRequest = ((ServletRequestAttributes) requestAttributes).getRequest();
            // 这个一般是Nginx反向代理设置的参数
            ip = httpServletRequest.getHeader("X-Real-IP");
            if (ip == null || ip.isEmpty() || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = httpServletRequest.getHeader("X-Forwarded-For");
            }
            if (ip == null || ip.isEmpty() || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = httpServletRequest.getHeader("Proxy-Client-IP");
            }
            if (ip == null || ip.isEmpty() || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = httpServletRequest.getHeader("WL-Proxy-Client-IP");
            }
            if (ip == null || ip.isEmpty() || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = httpServletRequest.getRemoteAddr();
            }
            if (IP_ADDRESS.contains(ip)) {
                ip = getIpAddress();
            }
        } else {
            ip = getIpAddress();
        }
        return ip;
    }

    /**
     * 获取本地ip地址
     *
     * @return 本地ip地址
     */
    private static String getIpAddress() {
        try {
            Enumeration<NetworkInterface> allNetInterfaces = NetworkInterface.getNetworkInterfaces();
            InetAddress ip;
            while (allNetInterfaces.hasMoreElements()) {
                NetworkInterface netInterface = allNetInterfaces.nextElement();
                if (!netInterface.isLoopback() && !netInterface.isVirtual() && netInterface.isUp()) {
                    Enumeration<InetAddress> addresses = netInterface.getInetAddresses();
                    while (addresses.hasMoreElements()) {
                        ip = addresses.nextElement();
                        if (ip instanceof Inet4Address) {
                            return ip.getHostAddress();
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * 解析xml(只针对解析标签下没有相同标签)
     *
     * @param xml  xml字符串
     * @param tags 多个节点用.号拼接
     */
    public static String analysisXmlStr(String xml, String tags) {
        try {
            // 将字符串转为XML
            Document doc = DocumentHelper.parseText(xml);
            // 获取根节点
            Element rootElt = doc.getRootElement();
            if (StrUtil.isNotBlank(tags)) {
                String[] split = tags.split("\\.");
                Element param = null;
                for (String s : split) {
                    if (Objects.isNull(param)) {
                        param = rootElt.element(s);
                    } else {
                        param = param.element(s);
                    }
                }
                if (!Objects.isNull(param)) {
                    return param.getText();
                }
            } else {
                return rootElt.getText();
            }
        } catch (DocumentException e) {
            e.printStackTrace();
            log.error("解析失败xml={}  tags={}", xml, tags);
        }
        return null;
    }

    /**
     * 获取操作系统,浏览器及浏览器版本信息
     *
     * @return Map
     */
    public static Map<String, String> getOsAndBrowserInfo() {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        Map<String, String> map = new HashMap<>(2);
        if (!Objects.isNull(requestAttributes)) {
            HttpServletRequest httpServletRequest = ((ServletRequestAttributes) requestAttributes).getRequest();
            String ua = httpServletRequest.getHeader("User-Agent");
            //转成UserAgent对象
            UserAgent userAgent = UserAgent.parseUserAgentString(ua);
            //获取浏览器信息
            Browser browser = userAgent.getBrowser();
            //获取系统信息
            OperatingSystem os = userAgent.getOperatingSystem();
            map.put("os", os.getName());
            map.put("browser", Objects.equals(browser.getName(), "Unknown") ? ua : browser.getName());
        }
        return map;
    }

    /**
     * 获取UUID字符串
     *
     * @return
     */
    public static String getUuidStr() {
        return UUID.randomUUID().toString();
    }

    /**
     * 获取创建时间
     *
     * @return
     */
    public static long getCurrentTime() {
        return System.currentTimeMillis() / 1000;
    }


    /**
     * 根据属性名设置属性值
     *
     * @param fieldName
     * @param object
     * @return
     */
    public static void setFieldValueByFieldName(Object object, String fieldName, Object value) {
        try {
            // 获取obj类的字节文件对象
            Class<?> c = object.getClass();
            Field field = getFieldInfo(c, c.getDeclaredFields(), fieldName);
            if (!Objects.isNull(field)) {
                // 取消语言访问检查
                field.setAccessible(true);
                // 给变量赋值
                field.set(object, value);
            }
        } catch (Exception e) {
            log.warn("根据属性名设置属性值 {} 失败", fieldName);
        }
    }

    /**
     * 根据属性名获取属性值
     *
     * @param fieldName
     * @param object
     * @return
     */
    public static Object getFieldValueByFieldName(String fieldName, Object object) {
        try {
            // 获取obj类的字节文件对象
            Class<?> c = object.getClass();
            Field field = getFieldInfo(c, c.getDeclaredFields(), fieldName);
            if (!Objects.isNull(field)) {
                //设置对象的访问权限，保证对private的属性的访问
                field.setAccessible(true);
                return field.get(object);
            }
        } catch (Exception e) {
            log.warn("根据属性名获取属性值 {} 失败", fieldName);
        }
        return null;
    }

    /**
     * 递归获取字段信息
     *
     * @param c
     * @param fields
     * @param fieldName
     * @return
     */
    private static Field getFieldInfo(Class<?> c, Field[] fields, Object fieldName) {
        for (Field field : fields) {
            if (Objects.equals(fieldName, field.getName())) {
                return field;
            }
        }
        Class<?> superclass = c.getSuperclass();
        if (Objects.isNull(superclass)) {
            return null;
        }
        Field[] declaredFields = superclass.getDeclaredFields();
        return getFieldInfo(superclass, declaredFields, fieldName);
    }
}
