package com.zero.utils.base;

import com.zero.utils.LogTag;
import com.zero.utils.PageResult;
import com.zero.utils.PublicResult;
import com.zero.utils.validation.group.Insert;
import com.zero.utils.validation.group.Update;
import io.swagger.annotations.ApiOperation;
import lombok.NoArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.List;
import java.util.Optional;

/**
 * 控制层基类
 *
 * @author zero
 */
@NoArgsConstructor
public class BaseController<E extends BaseEntity<ID>, ID, S extends IBase<E, ID, Q>, Q extends BaseQuery> {
    @Autowired
    protected S service;

    @GetMapping("get")
    @ApiOperation(value = "获取一条", notes = "通过主键获取一条信息")
    public PublicResult<Optional<E>> get(@RequestParam("id") ID id) {
        return service.get(id);
    }

    @GetMapping("getByIds")
    @ApiOperation(value = "获取多条", notes = "通过主键集合获取多条信息")
    public PublicResult<List<E>> getByIds(@RequestParam("ids") List<ID> ids) {
        return service.getByIds(ids);
    }

    @PostMapping("list")
    @ApiOperation(value = "列表", notes = "查询数据集合")
    public PublicResult<List<E>> list(@RequestBody @Validated Q query) {
        return service.list(query);
    }

    @PostMapping("pageList")
    @ApiOperation(value = "分页列表", notes = "查询分页数据集合")
    public PageResult<List<E>> pageList(@RequestBody @Validated Q query) {
        return service.pageList(query);
    }

    @PostMapping("del")
    @ApiOperation(value = "删除", notes = "公用删除接口")
    @LogTag
    public PublicResult deleteByIds(@RequestBody List<E> es) {
        return service.deleteByIds(es);
    }

    @PostMapping("add")
    @ApiOperation(value = "新增", notes = "公用新增接口")
    @LogTag
    public PublicResult save(@RequestBody @Validated({Insert.class}) E entity) {
        return service.add(entity);
    }

    @PostMapping("update")
    @ApiOperation(value = "修改", notes = "公用修改接口")
    @LogTag
    public PublicResult update(@RequestBody @Validated({Update.class}) E entity) {
        return service.update(entity);
    }

}
