package com.yunlongn.common.json;

import com.yunlongn.common.json.gson.GsonParser;
import com.yunlongn.common.json.jackson.JacksonParser;
import com.yunlongn.common.json.util.StringUtils;

import java.io.InputStream;
import java.lang.reflect.Type;
import java.util.List;

/**
 * json序列化适配器
 *
 * @author Yun 2020/04/26
 */
public class JsonUtils {

    private static JsonParser chooseJsonParser;

    static {
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        try {
            classLoader.loadClass("com.google.gson.GsonBuilder");
            chooseJsonParser = new GsonParser();
        } catch (ClassNotFoundException e) {
            try {
                classLoader.loadClass("com.fasterxml.jackson.databind.ObjectMapper");
                chooseJsonParser = new JacksonParser();
            } catch (ClassNotFoundException ex) {
                throw new RuntimeException("未找到任务json包，请先在当前项目的依赖配置文件中加入 gson或fackson");
            }
        }
    }

    public static <T> String toJsonString(T obj) {
        return toJsonString(obj, false, null);
    }

    public static <T> String toJsonString(T obj, boolean serializeNulls) {
        return toJsonString(obj, serializeNulls, null);
    }

    /**
     * 序列化
     *
     * @param obj
     * @param serializeNulls 是否需要序列化值为null的字段
     * @param <T>
     * @return
     */
    public static <T> String toJsonString(T obj, boolean serializeNulls, String datePattern) {
        if (obj instanceof String) {
            return (String) obj;
        }
        if (obj == null) {
            return null;
        }
        return chooseJsonParser.toJsonString(obj, serializeNulls, datePattern);
    }

    public static <T> T fromJson(String jsonStr, Class<T> tClass) {
        if (StringUtils.isNullOrEmpty(jsonStr)) {
            return null;
        }
        if (tClass == String.class) {
            return (T) jsonStr;
        }
        return chooseJsonParser.fromJson(jsonStr, tClass);
    }

    public static <T> T fromJson(InputStream jsonIn, Class<T> tClass) {
        return chooseJsonParser.fromJson(jsonIn, tClass);
    }

    public static <T> T fromJson(String jsonStr, Type type) {
        return chooseJsonParser.fromJson(jsonStr, type);
    }

    public static <T> T fromJson(InputStream jsonIn, Type type) {
        return chooseJsonParser.fromJson(jsonIn, type);
    }

    public static <T> List<T> fromJsonArray(String jsonStr, TypeReference<List<T>> typeReference) {
        if (StringUtils.isNullOrEmpty(jsonStr)) {
            return null;
        }
        return chooseJsonParser.fromJsonArray(jsonStr, typeReference);
    }

}
