package com.wong.support.http;

import com.alibaba.fastjson.JSONObject;
import com.wong.support.http.annotation.Header;
import com.wong.support.http.annotation.RequestBody;
import com.wong.support.http.annotation.Param;
import com.wong.support.http.properties.SupportClientProperties;
import com.wong.support.http.exception.IllegalParamException;
import com.wong.support.http.properties.SupportGlobalConfigProperties;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.net.URI;
import java.util.*;
import java.util.stream.Collectors;

/**
 * ClassName Context
 * Author chenwang
 * Date 2021/9/24 9:40
 * Description
 * Version 1.0
 */
@Slf4j
public class Context {

    public static Object handle(Method method, Object[] args,Configuration configuration){
        HttpClient client = HttpClientBuilder.create().build();
        SupportGlobalConfigProperties globalProperties = configuration.getBeanFactory().getBean(SupportGlobalConfigProperties.class);
        SupportClientProperties properties = configuration.getBeanFactory().getBean(SupportClientProperties.class);
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(properties.getConnectTimeout())
                .setConnectionRequestTimeout(properties.getRequestTimeout())
                .build();
        Map<String, Handler> handlerMap = configuration.getHandlerMap();
        Handler handler = handlerMap.get(method.getName());
        if(handler == null){
            throw new IllegalParamException("方法映射失败，请检查：method name:"+method.getName());
        }
        String domain = configuration.getDomain();
        Parameter[] parameters = method.getParameters();
        StringBuilder url = new StringBuilder(domain+handler.getUri());
        HttpRequestBase request;
        switch (handler.getMethod()){
            default:
            case GET:
                request = get(parameters,args,url);
                break;
            case POST:
                request = post(parameters,args,url);
                break;
        }
        request.setConfig(config);
        //默认的header加入。
        globalProperties.getGlobalHeader().forEach(request::addHeader);
        try {
            long l = System.currentTimeMillis();
            HttpResponse execute = client.execute(request);
            HttpEntity entity = execute.getEntity();
            String result = EntityUtils.toString(entity);
            int statusCode = execute.getStatusLine().getStatusCode();
            if (statusCode == HttpStatus.SC_OK) {
                // 返回json格式
                if(globalProperties.getLogPrint()) {
                    log.info("请求方式：{}",request.getMethod());
                    log.info("请求路径：{}",request.getURI());
                    log.info("返回结果：{}",result);
                    log.info("响应时间：{}ms",System.currentTimeMillis() - l);
                }
                if(checkBasicType(method.getReturnType())){
                    return result;
                }
                return JSONObject.parseObject(result, method.getReturnType());
            } else {
                log.warn("support http 请求响应状态码异常,请求路径：{}，状态码：{},相关信息：{}",request.getURI(), statusCode,execute.getStatusLine().getReasonPhrase());
            }
        } catch (IOException e) {
            log.error("请求/响应失败 msg:{}",e.getMessage());
            e.printStackTrace();
        }finally {
            HttpClientUtils.closeQuietly(client);
        }
        return null;
    }


    public static HttpRequestBase post(Parameter[] parameters, Object[] args, StringBuilder url) {
        HttpPost post = new HttpPost();
        Map<String,String> header = new HashMap<>();
        url.append("?");
        boolean hasBody = false;
        for (int i = 0; i < parameters.length; i++) {
            if(parameters[i].isAnnotationPresent(Header.class)){
                Header annotation = parameters[i].getAnnotation(Header.class);
                String headerName = annotation.value();
                if(!StringUtils.hasText(headerName)){
                    headerName = parameters[i].getName();
                }
                header.put(headerName, (String) args[i]);
            }else if (parameters[i].isAnnotationPresent(RequestBody.class)){
                checkHasBody(hasBody,parameters[i].getName());
                if(checkBasicType(args[i].getClass())){
                    throw new IllegalParamException("参数错误，参数对应注解不匹配，参数类型："+args[i].getClass());
                }
                url.deleteCharAt(url.length()-1);
                StringEntity s = new StringEntity(JSONObject.toJSONString(args[i]), "UTF-8");
                s.setContentType("application/json");
                post.setEntity(s);
                hasBody = true;
            }else if (parameters[i].isAnnotationPresent(Param.class)){
                checkHasBody(hasBody,parameters[i].getName());
                if(!checkBasicType(args[i].getClass())){
                    throw new IllegalParamException("参数错误，参数对应注解不匹配，参数类型："+args[i].getClass());
                }
                url.append(parameters[i].getName()).append("=");
                if (parameters[i] != null) {
                    url.append(parameters[i]);
                }
                url.append("&");
            }
        }
        header.forEach(post::addHeader);
        post.setURI(URI.create(url.toString()));
        log.info("POST 请求地址：{}",post.getEntity());
        return post;
    }

    public static HttpRequestBase get(Parameter[] parameters,Object[] args,StringBuilder url) {
        HttpGet get = new HttpGet();
        Map<String,String> header = new HashMap<>();
        url.append("?");
        boolean hasBody = false;
        for (int i = 0; i < parameters.length; i++) {
            if(parameters[i].isAnnotationPresent(Header.class)){
                Header annotation = parameters[i].getAnnotation(Header.class);
                String headerName = annotation.value();
                if(!StringUtils.hasText(headerName)){
                    headerName = parameters[i].getName();
                }
                header.put(headerName, (String) args[i]);
            }else if (parameters[i].isAnnotationPresent(RequestBody.class)){
                checkHasBody(hasBody,parameters[i].getName());
                if(checkBasicType(args[i].getClass())){
                    throw new IllegalParamException("参数错误，参数对应注解不匹配，参数类型："+args[i].getClass());
                }
                List<Field> fields = Arrays.asList(parameters[i].getClass().getDeclaredFields());
                fields.addAll(Arrays.asList(fields.getClass().getSuperclass().getDeclaredFields()));
                for (Field field : fields) {
                    field.setAccessible(true);
                    try {
                        Object o = field.get(args[i]);
                        if(null == o){
                            continue;
                        }
                        url.append(field.getName()).append("=").append(o).append("&");
                    } catch (IllegalAccessException e) {
                        e.printStackTrace();
                    }
                }
                hasBody = true;
            }else if (parameters[i].isAnnotationPresent(Param.class)){
                checkHasBody(hasBody,parameters[i].getName());
                if(!checkBasicType(args[i].getClass())){
                    throw new IllegalParamException("参数错误，参数对应注解不匹配，参数类型："+args[i].getClass());
                }
            }
        }
        header.forEach(get::addHeader);
        get.setURI(URI.create(url.toString()));
        log.info("GET 请求地址：{}",get.getURI());
        return get;
    }


    private static void checkHasBody(Boolean hasBody,String name){
        if(hasBody){
            throw new IllegalParamException("请求参数不符合规范！参数名："+name);
        }
    }

    public static Boolean checkBasicType(Class clazz){

        if (clazz.getSimpleName().equals("String")) {
            return true;
        } else if (clazz.getSimpleName().equals("Integer")) {
            return true;
        } else if (clazz.getSimpleName().equals("Long")) {
            return true;
        } else if (clazz.getSimpleName().equals("Double")) {
            return true;
        } else if (clazz.getSimpleName().equals("Float")) {
            return true;
        } else {
            return clazz.getSimpleName().equals("Short");
        }
    }

}
