package com.hsh.tangramlibrary.widget

import android.content.Context
import android.graphics.Color
import android.text.TextUtils
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.TextView
import com.hsh.tangramlibrary.bean.CustomTitleCellBean
import com.hsh.tangramlibrary.utils.Utils
import com.tmall.wireless.tangram.structure.BaseCell
import com.tmall.wireless.tangram.structure.CellRender
import com.tmall.wireless.tangram.structure.view.ITangramViewLifeCycle
import com.tmall.wireless.tangram.util.ImageUtils

/**
 *带有标题和刷新，都支持设置文字和图片。如果标题设置了文字，则图片隐藏，没有设置文字则使用图片，刷新相同。
 * Created by yuanye on 2018/1/12.
 */
class CustomTitleView(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0) :
        ITangramViewLifeCycle, FrameLayout(context, attrs, defStyleAttr) {
    constructor(context: Context, attrs: AttributeSet? = null) : this(context, attrs, 0)
    constructor(context: Context) : this(context, null)

    companion object {
        val KEY_TITLE_BG_IMG_STRING = "imgUrl"
        val KEY_ITEMS = "items"
        val KEY_ITEM_TYPE = "ComponentSubType"
    }

    var titleImage: ImageView = ImageView(context)
    var titleText: TextView = TextView(context)
    var refreshText: TextView = TextView(context)
    var refreshImage: ImageView = ImageView(context)

    init {
        val titleLayoutParams = FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.MATCH_PARENT)
        titleText.layoutParams = titleLayoutParams
        titleText.ellipsize = TextUtils.TruncateAt.END
        titleText.gravity = Gravity.CENTER_VERTICAL

        val padding = Utils.dip2px(context, 20)
        val refreshTextLayoutParams = FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.MATCH_PARENT)
        refreshTextLayoutParams.gravity = Gravity.CENTER_VERTICAL.or(Gravity.RIGHT)
        refreshText.layoutParams = refreshTextLayoutParams
        refreshText.ellipsize = TextUtils.TruncateAt.END
        refreshText.gravity = Gravity.CENTER_VERTICAL
        refreshText.setPadding(padding, padding / 8, 0, padding / 8)

        val titleImgLayoutParams = FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.MATCH_PARENT)
        titleImage.layoutParams = titleImgLayoutParams
        titleImage.scaleType = ImageView.ScaleType.CENTER

        val refreshImgLayoutParams = FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.MATCH_PARENT)
        refreshImgLayoutParams.gravity = Gravity.CENTER_VERTICAL.or(Gravity.RIGHT)
        refreshImage.layoutParams = refreshImgLayoutParams
        refreshImage.scaleType = ImageView.ScaleType.CENTER
        refreshImage.setPadding(padding, padding / 8, 0, padding / 8)

        addView(titleImage)
        addView(titleText)
        addView(refreshImage)
        addView(refreshText)
    }

    @CellRender
    override fun postBindView(cell: BaseCell<*>?) {
        val leftFlag = cell?.optBoolParam(CustomTitleCellBean.KEY_TITLE_LEFT_FLAG)
        val gravity = if (leftFlag == true) {
            Gravity.CENTER_VERTICAL.or(Gravity.LEFT)
        } else {
            Gravity.CENTER
        }
        var layoutParams: FrameLayout.LayoutParams? = titleImage.layoutParams as? LayoutParams
        layoutParams?.gravity = gravity
        layoutParams = titleText.layoutParams as? LayoutParams
        layoutParams?.gravity = gravity

        //如果设置了标题文字，则不显示标题图片，反之则显示标题图片
        val titleTextString = cell?.optStringParam(CustomTitleCellBean.KEY_TITLE + CustomTitleCellBean.KEY_TEXT)
        if (titleTextString?.isNotEmpty() == true) {
            titleText.visibility = View.VISIBLE
            titleImage.visibility = View.GONE
            bindText(cell, titleText)
        } else {
            titleText.visibility = View.GONE
            titleImage.visibility = View.VISIBLE
            bindImageView(cell, titleImage)
        }

        //如果设置了刷新文字，则不显示刷新图片，反之则显示刷新图片
        val refreshTextString = cell?.optStringParam(CustomTitleCellBean.KEY_REFRESH + CustomTitleCellBean.KEY_TEXT)
        if (refreshTextString?.isNotEmpty() == true) {
            refreshText.visibility = View.VISIBLE
            refreshImage.visibility = View.GONE
            bindText(cell, refreshText)
        } else {
            refreshText.visibility = View.GONE
            refreshImage.visibility = View.VISIBLE
            bindImageView(cell, refreshImage)
        }
        setOnClickListener(cell)
    }

    private fun bindText(cell: BaseCell<*>?, textView: TextView) {
        val textKey = when (textView) {
            titleText -> CustomTitleCellBean.KEY_TITLE
            refreshText -> CustomTitleCellBean.KEY_REFRESH
            else -> ""
        }
        val text = cell?.optStringParam(textKey + CustomTitleCellBean.KEY_TEXT)
        //标题设置
        if (text?.isNotEmpty() == true) {
            textView.visibility = View.VISIBLE
            textView.text = text
            val textColor = cell.optIntParam(textKey + CustomTitleCellBean.KEY_TEXT_COLOR)
            if (textColor != 0) {
                textView.setTextColor(textColor)
            } else {
                textView.setTextColor(Color.BLACK)
            }
            val textSize = cell.optIntParam(textKey + CustomTitleCellBean.KEY_TEXT_SIZE)
            if (textSize > 0) {
                textView.textSize = textSize.toFloat()
            } else {
                textView.textSize = 12f
            }
            val textMaxLine = cell.optIntParam(textKey + CustomTitleCellBean.KEY_TEXT_MAX_LINE)
            if (textMaxLine > 0) {
                textView.maxLines = textMaxLine
            } else {
                textView.maxLines = 1
            }
            var margin = cell.optIntParam(textKey + CustomTitleCellBean.KEY_TEXT_MARGIN)
            if (margin < 0) {
                margin = 0
            }
            val layoutParams: MarginLayoutParams = textView.layoutParams as MarginLayoutParams
            val px = Utils.dip2px(context, margin.toFloat())
            layoutParams.topMargin = px
            layoutParams.leftMargin = px
            layoutParams.rightMargin = px
            layoutParams.bottomMargin = px
        } else {
            textView.visibility = View.GONE
        }
    }

    private fun bindImageView(cell: BaseCell<*>?, imgView: ImageView) {
        val imgKey = when (imgView) {
            titleImage -> CustomTitleCellBean.KEY_TITLE
            refreshImage -> CustomTitleCellBean.KEY_REFRESH
            else -> ""
        }
        val imgUrl = cell?.optStringParam(imgKey + CustomTitleCellBean.KEY_IMG_URL)
        if (imgUrl?.isNotEmpty() == true) {
            imgView.visibility = View.VISIBLE
            ImageUtils.doLoadImageUrl(imgView, imgUrl)
        } else {
            imgView.visibility = View.GONE
        }
        val imgMargin = Utils.dip2px(context, cell?.optIntParam(imgKey + CustomTitleCellBean.KEY_IMG_MARGIN)
                ?: 0)
        val layoutParams: MarginLayoutParams = imgView.layoutParams as MarginLayoutParams
        layoutParams.bottomMargin = imgMargin
        layoutParams.topMargin = imgMargin
        layoutParams.leftMargin = imgMargin
        layoutParams.rightMargin = imgMargin
    }

    @CellRender
    override fun postUnBindView(cell: BaseCell<*>?) {
        ImageUtils.doLoadImageUrl(titleImage, "")
        ImageUtils.doLoadImageUrl(refreshImage, "")
    }

    @CellRender
    override fun cellInited(cell: BaseCell<*>?) {
        setOnClickListener(cell)
    }

    override fun setOnClickListener(l: OnClickListener?) {
        if (refreshImage.visibility == View.VISIBLE ||
                refreshText.visibility == View.VISIBLE) {
            refreshImage.setOnClickListener(l)
            refreshText.setOnClickListener(l)
        } else {
            super.setOnClickListener(l)
        }
    }
}