package com.solarterms.hakuro.web.http;


import com.solarterms.hakuro.framework.core.bean.Log;
import com.solarterms.hakuro.framework.utils.collections.CollectionUtils;
import com.solarterms.hakuro.framework.utils.function.StringUtils;
import com.solarterms.hakuro.framework.utils.converter.BeanUtils;
import com.solarterms.hakuro.framework.utils.converter.JsonUtils;
import com.solarterms.hakuro.framework.utils.log.LogUtils;
import com.solarterms.hakuro.web.context.util.ApplicationContextUtils;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.RestTemplate;

import java.net.MalformedURLException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;


/**
 * @author Created by Alan on 2021/5/19.
 */
@SuppressWarnings("unchecked")
public class RestTemplateUtils {

    public static <T> ResponseEntity<T> postForText(String url, String data, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            HttpEntity<String> entity = generateTextEntity(data);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, data, e);
        }
        return responseEntity;
    }

    public static <T> ResponseEntity<T> postForJson(String url, Object object, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            HttpEntity<String> entity = generateJsonEntity(object);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, object, e);
        }
        return responseEntity;
    }

    public static <T> ResponseEntity<T> postForJson(String url, Object object, Class<T> responseType, Map<String,String> headers) {
        ResponseEntity<T> responseEntity;
        try {
            HttpEntity<String> entity = generateJsonEntity(object,headers);
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity);
        } catch (Exception e) {
            responseEntity = handleException(url, object, e);
        }
        return responseEntity;
    }

//    public static <T> ResponseEntity<T> postForJson(String url, String jsonString, Class<T> responseType) {
//        ResponseEntity<T> responseEntity;
//        try {
//            HttpEntity<String> entity = generateJsonEntity(jsonString);
//            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
//            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity);
//        } catch (Exception e) {
//            responseEntity = handleException(url, jsonString, e);
//        }
//        return responseEntity;
//    }

    public static <T> ResponseEntity<T> postForJson(String url, Object object, ParameterizedTypeReference<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            HttpEntity<String> entity = generateJsonEntity(object);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, object, e);
        }
        return responseEntity;
    }


//    public static <T> ResponseEntity<T> postForJson(String url, String jsonString, ParameterizedTypeReference<T> responseType) {
//        ResponseEntity<T> responseEntity;
//        try {
//            HttpEntity<String> entity = generateJsonEntity(jsonString);
//            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
//            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity);
//        } catch (Exception e) {
//            responseEntity = handleException(url, jsonString, e);
//        }
//        return responseEntity;
//    }

    public static <T> ResponseEntity<T> postForForm(String url, Object object, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
            MultiValueMap<String, String> map = new LinkedMultiValueMap<>();
            map.setAll(BeanUtils.convert(object, Map.class));
            HttpEntity<MultiValueMap<String, String>> entity = new HttpEntity<>(map, headers);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, object, e);
        }
        return responseEntity;
    }

    public static <T> ResponseEntity<T> postForMap(String url, Map mapParam, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
            MultiValueMap<String, String> map = new LinkedMultiValueMap<>();
            map.setAll(mapParam);
            HttpEntity<MultiValueMap<String, String>> entity = new HttpEntity<>(map, headers);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, mapParam, e);
        }
        return responseEntity;
    }

    public static <T> ResponseEntity<T> postForFile(String url, byte[] file, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            ByteArrayResource bar = new ByteArrayResource(file);
            HttpEntity<ByteArrayResource> entity = new HttpEntity<>(bar);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, "[file]", e);
        }
        return responseEntity;
    }

    public static <T> ResponseEntity<T> postForMultiPart(String url, MultiValueMap<String, Object> mapParam, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.MULTIPART_FORM_DATA);
            HttpEntity<MultiValueMap<String, Object>> entity = new HttpEntity<>(mapParam, headers);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, mapParam, e);
        }
        return responseEntity;
    }

    public static <T> ResponseEntity<T> postForMultiPart(String url, MultiValueMap<String, Object> mapParam, HttpHeaders headers, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            HttpEntity<MultiValueMap<String, Object>> entity = new HttpEntity<>(mapParam, headers);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).exchange(url, HttpMethod.POST, entity, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", entity, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, mapParam, e);
        }
        return responseEntity;
    }

    @SuppressWarnings("unchecked")
    public static <T> ResponseEntity<T> getForEntity(String url, Object object, Class<T> responseType) {
        Map<String, String> params = BeanUtils.convert(object, LinkedHashMap.class);
        return getForEntity(url, params, responseType);
    }

    public static <T> ResponseEntity<T> getForEntity(String url, Map<String, String> params, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            String generateUrl = createUrlWithParameters(url, params);
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).getForEntity(generateUrl, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：", params, "；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, params, e);
        }
        return responseEntity;
    }

    private static HttpEntity<String> generateJsonEntity(Object object, Map<String, String> headerParams) {
        String jsonString;
        if (object instanceof String) {
            jsonString = (String) object;
        } else {
            jsonString = JsonUtils.toNonNullJsonString(object);
        }
        HttpHeaders headers = new HttpHeaders();
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("Accept", MediaType.APPLICATION_JSON.toString());
        headerParams.forEach(headers::add);
        return new HttpEntity<>(jsonString, headers);
    }

    public static <T> ResponseEntity<T> getForEntity(String url, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).getForEntity(url, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", url, "；请求：null；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(url, null, e);
        }
        return responseEntity;
    }

    public static <T> ResponseEntity<T> getForEntity(URI uri, Class<T> responseType) {
        ResponseEntity<T> responseEntity;
        try {
            long start = System.currentTimeMillis();
            responseEntity = ((RestTemplate) ApplicationContextUtils.getBean("restTemplate")).getForEntity(uri, responseType);
            long end = System.currentTimeMillis();
            LogUtils.info(Log.REST_CONNECTION_LOGGER, "http请求，url：", uri.getPath(), "；请求：null；响应：", responseEntity, "；耗时：", end - start);
        } catch (Exception e) {
            responseEntity = handleException(uri.toString(), null, e);
        }
        return responseEntity;
    }

    private static HttpEntity<String> generateTextEntity(String data) {
        HttpHeaders headers = new HttpHeaders();
        MediaType type = MediaType.parseMediaType("text/plain; charset=UTF-8");
        headers.setContentType(type);
        headers.add("Accept", MediaType.APPLICATION_JSON.toString());
        return new HttpEntity<>(data, headers);
    }

    private static HttpEntity<String> generateJsonEntity(Object object) {
        String jsonString;
        if (object instanceof String) {
            jsonString = (String) object;
        } else {
            jsonString = JsonUtils.toNonNullJsonString(object);
        }
        HttpHeaders headers = new HttpHeaders();
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("Accept", MediaType.APPLICATION_JSON.toString());
        return new HttpEntity<>(jsonString, headers);
    }

//    private static HttpEntity<String> generateJsonEntity(String jsonString) {
//        HttpHeaders headers = new HttpHeaders();
//        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
//        headers.setContentType(type);
//        headers.add("Accept", MediaType.APPLICATION_JSON.toString());
//        return new HttpEntity<>(jsonString, headers);
//    }

    private static ResponseEntity handleException(String url, Object requestBody, Exception e) {
        ResponseEntity responseEntity;
        if (e instanceof HttpStatusCodeException) {
            responseEntity = new ResponseEntity<>(((HttpStatusCodeException) e).getResponseBodyAsString(), ((HttpStatusCodeException) e).getStatusCode());
        } else if (e.getCause() instanceof SocketTimeoutException) {
            responseEntity = new ResponseEntity<>(HttpStatus.REQUEST_TIMEOUT);
        } else {
            responseEntity = new ResponseEntity<>(HttpStatus.SERVICE_UNAVAILABLE);
        }
        LogUtils.error(Log.REST_CONNECTION_LOGGER, "http请求发生异常，url：", url, "；请求：", JsonUtils.toNonNullJsonString(requestBody), "；响应：", responseEntity.getBody(), "；异常：", e.getMessage());
        String message = String.format("%s发生异常请求，响应码：%s", getHostUrl(url), responseEntity.getStatusCode());
        return responseEntity;
    }

    private static String getHostUrl(String link){
        String hostUrl = link;
        try {
            URL url = new URL(link);
            hostUrl = String.format("%s://%s%s:%d",url.getProtocol(),url.getHost(),url.getPath(),url.getPort());
        } catch (MalformedURLException e) {
            e.printStackTrace();
        }
        return hostUrl;
    }

    public static String createUrlWithParameters(String url, Map<String, String> parameters) {
        StringBuffer sb = (new StringBuffer(url)).append("?");
        parameters.forEach((k, v) -> {
            if (StringUtils.isNotBlank(k) && StringUtils.isNotEmpty(v)) {
                sb.append(k).append("=").append(v).append("&");
            }
        });
        return sb.substring(0, sb.length() - 1);
    }

    public static String createUrlWithParameters(String url, Object object) {
        Map<String, String> params = BeanUtils.convert(object, Map.class);
        return createUrlWithParameters(url, params);
    }

    public static String createUrlWithSortedParameters(String url, LinkedHashMap<String, String> parameters) {
        List<Map.Entry<String, String>> paramList = new ArrayList<>(parameters.entrySet());
        CollectionUtils.sort(paramList, Comparator.comparing(o -> (o.getKey())));
        StringBuilder buf = new StringBuilder();
        for (Map.Entry<String, String> item : paramList) {
            if (StringUtils.isNotBlank(item.getKey()) && StringUtils.isNotEmpty(item.getValue())) {
                buf.append(item.getKey()).append("=").append(item.getValue()).append("&");
            }
        }
        StringBuilder resultBuffer = new StringBuilder().append(url).append("?").append(buf.toString());
        return resultBuffer.deleteCharAt(resultBuffer.length() - 1).toString();
    }

    public static String createUrlWithSortedParameters(String url, Object object) {
        LinkedHashMap<String, String> params = BeanUtils.convert(object, LinkedHashMap.class);
        return createUrlWithSortedParameters(url, params);
    }

    public static String createSortedParameters(Object object) {
        LinkedHashMap<String, String> params = BeanUtils.convert(object, LinkedHashMap.class);
        return createSortedParameters(params);
    }

    public static String createSortedParameters(LinkedHashMap<String, String> parameters) {
        List<Map.Entry<String, String>> paramList = new ArrayList<>(parameters.entrySet());
        CollectionUtils.sort(paramList, Comparator.comparing(o -> (o.getKey())));
        StringBuilder buf = new StringBuilder();
        for (Map.Entry<String, String> item : paramList) {
            if (StringUtils.isNotBlank(item.getKey()) && StringUtils.isNotEmpty(item.getValue())) {
                buf.append(item.getKey()).append("=").append(item.getValue()).append("&");
            }
        }
        StringBuilder resultBuffer = new StringBuilder().append(buf.toString());
        return resultBuffer.deleteCharAt(resultBuffer.length() - 1).toString();
    }

    public static void main(String[] args) {
        LinkedHashMap<String, String> params = new LinkedHashMap<>();
        params.put("versionNo", "1.03");
        params.put("channelId", "10000000012");
        params.put("merchId", "1000010");
        params.put("identityRole", "1");
        params.put("userID", "111_130124199104050015");
        String result = createUrlWithSortedParameters("", params);
        System.out.println(result);
    }
}
