package com.solarterms.hakuro.web.common;

import com.solarterms.hakuro.framework.utils.function.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import javax.servlet.http.HttpServletRequest;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Enumeration;
import java.util.Objects;
/**
 * @author Created by Alan on 2021/5/19.
 */
public class IpAddressUtils {

    public static final String X_FORWARDED_FOR = "x-forwarded-for";
    private static final String UNKNOWN_STRING = "unknown";

    public static String getHostAddress() {
        try {
            return Objects.requireNonNull(getLocalHostAddress()).getHostAddress();
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    public static String getClientAddress(HttpServletRequest request) {
        return getClientAddressByRealIp(request, X_FORWARDED_FOR);
    }

    private static String getClientAddressByRealIp(HttpServletRequest request, String addressType) {
        String ip = request.getHeader(addressType);
        if (StringUtils.isBlank(ip) || UNKNOWN_STRING.equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }

        if (StringUtils.isBlank(ip) || UNKNOWN_STRING.equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }

        if (StringUtils.isBlank(ip) || UNKNOWN_STRING.equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }

        if (ip != null && ip.length() > 15) {
            if (ip.indexOf(",") > 0) {
                ip = ip.substring(0, ip.indexOf(","));
            }
        }
        return ip;
    }

    private static InetAddress getLocalHostAddress() {
        try {
            InetAddress candidateAddress = null;
            // 遍历所有的网络接口
            for (Enumeration interfaces = NetworkInterface.getNetworkInterfaces(); interfaces.hasMoreElements(); ) {
                NetworkInterface networkInterface = (NetworkInterface) interfaces.nextElement();
                // 在所有的接口下再遍历IP
                for (Enumeration addresses = networkInterface.getInetAddresses(); addresses.hasMoreElements(); ) {
                    InetAddress address = (InetAddress) addresses.nextElement();
                    // 排除loopback类型地址
                    if (!address.isLoopbackAddress()) {
                        if (address.getHostAddress().contains(".")) {
                            String prefix = address.getHostAddress().substring(0, address.getHostAddress().indexOf("."));
                            if (NumberUtils.isParsable(prefix)) {
                                // 包含.分隔符，为测试或是线上环境
                                return address;
                            }
                        } else if (candidateAddress == null) {
                            // 不包含.分隔符，先记录候选地址
                            candidateAddress = address;
                        }
                    }
                }
            }
            if (candidateAddress != null) {
                return candidateAddress;
            }
            // 如果没有发现 non-loopback地址.只能用最次选的方案
            InetAddress jdkSuppliedAddress = InetAddress.getLocalHost();
            return jdkSuppliedAddress;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public static void main(String[] args) {
        System.out.println(IpAddressUtils.getLocalHostAddress());
    }
}
