package com.solarterms.hakuro.web.data.redis;

import com.solarterms.hakuro.web.context.util.ApplicationContextUtils;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.Assert;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

@SuppressWarnings("unchecked")
public abstract class RedisClient {

    private static RedisTemplate getRedisTemplate() {
        RedisTemplate redisTemplate = ((RedisTemplate) ApplicationContextUtils.getBean("redisTemplate"));
        Assert.notNull(redisTemplate, "获取RedisTemplate对象失败，请检查redis相关依赖和配置文件是否正常");
        return redisTemplate;
    }

    /**
     * 放入普通对象
     */
    public static void set(Object key, Object value) {
        getRedisTemplate().opsForValue().set(key, value);
    }

    /**
     * 放入普通对象，并设置过期时间
     */
    public static void set(Object key, Object value, long expire, TimeUnit timeUnit) {
        getRedisTemplate().opsForValue().set(key, value, expire, timeUnit);
    }

    /**
     * 设置特定缓存失效时间
     */
    public static Boolean expire(Object key, long expire, TimeUnit timeUnit) {
        return getRedisTemplate().expire(key, expire, timeUnit);
    }

    /**
     * 设置特定缓存失效时间,如果存在key则不覆盖
     */
    public static Boolean expireIfInexistence(Object key, long expire, TimeUnit timeUnit) {
        return getRedisTemplate().hasKey(key) ? true : getRedisTemplate().expire(key, expire, timeUnit);
    }

    /**
     * 设置特定缓存失效时间
     */
    public static Boolean expireAt(Object key, Date expireAt) {
        return getRedisTemplate().expireAt(key, expireAt);
    }

    /**
     * 获取普通对象
     */
    public static Object get(Object key) {
        return getRedisTemplate().opsForValue().get(key);
    }

    /**
     * 检查是否存在缓存对象
     */
    public static boolean exists(Object key) {
        return getRedisTemplate().hasKey(key);
    }

    /**
     * 删除普通对象
     */
    public static boolean delete(Object key) {
        getRedisTemplate().delete(key);
        return !exists(key);
    }

    /**
     * 读取Map
     */
    public static Map<Object, Object> entries(Object key) {
        return getRedisTemplate().opsForHash().entries(key);
    }

    /**
     * 放一个键值对
     */
    public static void put(Object key, Object k, Object v) {
        getRedisTemplate().opsForHash().put(key, k, v);
    }


    /**
     * 放入Map
     */
    public static void putAll(Object key, Map<?, ?> map) {
        RedisTemplate redisTemplate = getRedisTemplate();
        if (map.isEmpty()) {
            return;
        }
        byte[] rawKey = redisTemplate.getKeySerializer().serialize(key);
        Map<byte[], byte[]> hashes;
        if (map instanceof ConcurrentHashMap) {
            hashes = new ConcurrentHashMap<>(map.size());
        } else {
            hashes = new LinkedHashMap<>(map.size());
        }
        for (Map.Entry entry : map.entrySet()) {
            hashes.put(redisTemplate.getHashKeySerializer().serialize(entry.getKey()),
                    redisTemplate.getHashValueSerializer().serialize(entry.getValue()));
        }
        redisTemplate.execute(connection -> {
            connection.hMSet(rawKey, hashes);
            return null;
        }, true);
    }

    /**
     * 缓存对象值递增
     */
    public static Long increment(Object key) {
        return getRedisTemplate().opsForValue().increment(key, 1);
    }

    /**
     * 缓存对象增加特定值
     */
    public static Long increment(Object key, Long k) {
        return getRedisTemplate().opsForValue().increment(key, k);
    }

    /**
     * 缓存对象值递减
     */
    public static Long decrease(Object key) {
        return getRedisTemplate().opsForValue().increment(key, -1);
    }

    /**
     * 批量插入
     */
    public static Integer pipeLinedSet(Map<Object, Object> entries) {
        RedisTemplate redisTemplate = getRedisTemplate();
        return redisTemplate.executePipelined((RedisCallback<Object>) connection -> {
            for (Map.Entry<Object, Object> entry : entries.entrySet()) {
                connection.set(redisTemplate.getKeySerializer().serialize(entry.getKey()), redisTemplate.getValueSerializer().serialize(entry.getValue()));
            }
            return null;
        }).size();
    }

    /**
     * 批量获取
     */
    public static Map<Object, Object> pipeLinedGet(Set<Object> keys) {
        Object[] keysArray = keys.toArray();
        RedisTemplate redisTemplate = getRedisTemplate();
        List<Object> resultList = redisTemplate.executePipelined((RedisCallback<List<Object>>) connection -> {
            for (Object o : keysArray) {
                connection.get(redisTemplate.getKeySerializer().serialize(o));
            }
            return null;
        });

        Map<Object, Object> resultMap = new HashMap<>();
        for (int i = 0; i < keysArray.length; i++) {
            if (null == resultList.get(i)) {
                continue;
            }
            resultMap.put(keysArray[i], resultList.get(i));
        }
        return resultMap;
    }

    /**
     * 获取特定格式的key值
     */
    public static Set<Object> getKeysByPattern(String pattern) {
        return getRedisTemplate().keys(pattern);
    }

    /**
     * 通过特定格式匹配删除缓存
     */
    public static void deleteByPattern(String pattern) {
        RedisTemplate redisTemplate = getRedisTemplate();
        Set<String> keys = redisTemplate.keys(pattern);
        redisTemplate.delete(keys);
    }

    /**
     * 分布式锁
     */
    public static boolean setNX(Object key, Object value) {
        return getRedisTemplate().opsForValue().setIfAbsent(key, value);
    }

    /**
     * 分布式锁-带过期时间
     */
    public static boolean setNX(Object key, Object value, long expire, TimeUnit timeUnit) {
        RedisTemplate redisTemplate = getRedisTemplate();
        Boolean result = (Boolean) redisTemplate.execute((RedisCallback<Boolean>) connection -> {
            Object obj = connection.execute("set",
                    redisTemplate.getKeySerializer().serialize(key), redisTemplate.getValueSerializer().serialize(value),
                    "NX".getBytes(), "EX".getBytes(), redisTemplate.getValueSerializer().serialize(timeUnit.toSeconds(expire)));
            return obj != null;
        });
        return result == null ? false : result;
    }

    /**
     * 有效时间-小时
     */
    public static int getExpireHours(Object key) {
        return getRedisTemplate().getExpire(key, TimeUnit.HOURS).intValue();
    }

    /**RedisUtils
     * 有效时间-分钟
     */
    public static Long getExpireMinutes(Object key) {
        return getRedisTemplate().getExpire(key, TimeUnit.MINUTES);
    }

    /**
     * 哈希获取大key所有，返回map
     */
    public static Map hGetAll(Object key) {
        return getRedisTemplate().opsForHash().entries(key);
    }

    /**
     * 哈希获取
     */
    public static Object hGet(Object key, Object hashKey) {
        return getRedisTemplate().opsForHash().get(key, hashKey);
    }

    /**
     * 哈希查key
     */
    public static boolean hExists(Object key, Object hashKey) {
        return getRedisTemplate().opsForHash().hasKey(key, hashKey);
    }

    /**
     * 哈希删除key
     */
    public static Long hDelete(Object key, Object hashKey) {
        return getRedisTemplate().opsForHash().delete(key, hashKey);
    }

    /**
     * 插入一个队列元素，如果队列不存在则创建
     */
    public static Long lpush(Object key, Object value) {
        return getRedisTemplate().opsForList().leftPush(key, value);
    }

    /**
     * 插入一批队列元素，如果队列不存在则创建
     */
    public static Long lpushAll(Object key, Object... values) {
        return getRedisTemplate().opsForList().leftPushAll(key, values);
    }

    /**
     * 插入一批队列元素，如果队列不存在则创建
     */
    public static Long lpushAll(Object key, Collection<Object> values) {
        return getRedisTemplate().opsForList().leftPushAll(key, values);
    }

    /**
     * 弹出一个队列元素，如果没有数据返回null
     */
    public static Object rpop(Object key) {
        return getRedisTemplate().opsForList().rightPop(key);
    }

    /**
     * 弹出一个队列元素，如果没有数据则阻塞，直到指定的超时时间返回null
     */
    public static Object brpop(Object key, long timeout, TimeUnit timeUnit) {
        RedisTemplate redisTemplate = getRedisTemplate();
        List<byte[]> pipeLinedResult = redisTemplate.executePipelined((RedisCallback<List<byte[]>>) connection -> {
            List<byte[]> r = connection.bRPop((int) timeUnit.toSeconds(timeout), redisTemplate.getKeySerializer().serialize(key));
            return r == null || r.isEmpty() ? null : r;
        });
        if (pipeLinedResult.isEmpty() || pipeLinedResult.get(0) == null) {
            return null;
        }
        return ((List) (Object) pipeLinedResult.get(0)).get(1);
    }

    /**
     * 发布消息
     */
    public static void publish(String channel, Object message) {
        getRedisTemplate().convertAndSend(channel, message);
    }

}
