package oth.syk.common.tools;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTCreationException;
import com.auth0.jwt.exceptions.TokenExpiredException;
import com.auth0.jwt.interfaces.DecodedJWT;

import java.io.UnsupportedEncodingException;
import java.time.Instant;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * app登录token的生成和解析
 */
public class JwtUtils1 {
    /**
     * token秘钥，请勿泄露，请勿随便修改
     */
    public static final String SECRET = "gogorealtor";
    /**
     * token 过期时间: 10天
     */
    public static final int tokenDurationUnit = Calendar.DATE;
    public static final int tokenRefreshDurationUnit = Calendar.DATE;
    public static final int tokenDuration = 365;
    public static final int tokenRefreshDuration = 15;
    
    private static final String pointKey = "userId";
    private static final String authoritiesKey = "authorities";
    private static final String fooKey = "foo";

    /**
     * JWT生成Token.<br/>
     * <p>
     * JWT构成: header, payload, signature
     *
     * @throws UnsupportedEncodingException 
     * @throws JWTCreationException 
     * @throws IllegalArgumentException 
     */
    public static String createToken(Long point) throws IllegalArgumentException, JWTCreationException, UnsupportedEncodingException {
        Date iatDate = new Date();
        // expire time
        Calendar nowTime = Calendar.getInstance();
        nowTime.add(tokenDurationUnit, tokenDuration);
        Date expiresDate = nowTime.getTime();

        // header Map
        Map<String, Object> map = new HashMap<>();
        map.put("alg", "HS256");
        map.put("typ", "JWT");

        // build token
        // param backups {iss:Service, aud:APP}
        String token = JWT.create().withHeader(map) // header
                .withClaim("iss", "jwwd") // payload
                .withClaim("aud", "APP")
                .withClaim(pointKey, point)
                .withIssuedAt(iatDate) // sign time
                .withExpiresAt(expiresDate) // expire time
                .sign(Algorithm.HMAC256(SECRET)); // signature

        return token;
    }
    
    public static String createToken(Long point, String[] authorities) throws IllegalArgumentException, JWTCreationException, UnsupportedEncodingException {
        Date iatDate = new Date();
        // expire time
        Calendar nowTime = Calendar.getInstance();
        nowTime.add(tokenDurationUnit, tokenDuration);
        Date expiresDate = nowTime.getTime();

        // header Map
        Map<String, Object> map = new HashMap<>();
        map.put("alg", "HS256");
        map.put("typ", "JWT");

        // build token
        // param backups {iss:Service, aud:APP}
        String token = JWT.create().withHeader(map) // header
                .withClaim("iss", "jwwd") // payload
                .withClaim("aud", "APP")
                .withClaim(pointKey, point)
                .withArrayClaim("authorities", authorities)
                .withIssuedAt(iatDate) // sign time
                .withExpiresAt(expiresDate) // expire time
                .sign(Algorithm.HMAC256(SECRET)); // signature

        return token;
    }

    /**
     * 解密Token
     *
     * @param token
     * @return
     * @throws Exception
     */
    public static DecodedJWT verifyToken(String token) throws Exception {
        JWTVerifier verifier = JWT.require(Algorithm.HMAC256(SECRET)).build();
        return verifier.verify(token);
    }
    
    public static String getRealToken(String token) {
    	String headerStart = ConfigUtils1.getPropertyValue("authorization.header.start");
    	if(!token.startsWith(headerStart)) {
    		throw new IllegalArgumentException("token开头错误！");
    	}
    	String authToken = token.substring(headerStart.length()); // The part after "Bearer "
    	return authToken;
	}

    /**
     * 根据Token获取user_id
     *
     * @param token
     * @return user_id
     */
    public static Long getPoint(String token) throws Exception {
    	Long point = JWT.decode(token).getClaim(pointKey).asLong();
        return point;
    }
    
    public static String getPoint(DecodedJWT decodedJWT) throws Exception {
        String point = decodedJWT.getClaim(pointKey).asString();
        return point;
    }
    
    public static String[] getAuthorities(DecodedJWT decodedJWT) throws Exception {
    	String[] authorities = decodedJWT.getClaim(authoritiesKey).asArray(String.class);
    	return authorities;
    }
    
    public static String getFoo(String token) throws Exception {
    	String foo = JWT.decode(token).getClaim(fooKey).asString();
        return foo;
    }
    
    /**
     * token 刷新：
     * 1.小于TIME_OUT直接通过；
     * 2.大于TIME_OUT 小于FORBID_REFRES_HTIME需要刷新；
     * 3.超过FORBID_REFRES_HTIME 直接返回禁用刷新；
     *
     * @param oldToken
     * @return
     * @throws UnsupportedEncodingException 
     * @throws IllegalArgumentException 
     */
    public static String refresh(String oldToken) throws IllegalArgumentException, UnsupportedEncodingException {
        long tokenDurationTime = TimeUnit.SECONDS.convert(tokenDuration, TimeUnit.DAYS);//token持续时间/分钟
        long tokenRefreshDurationTime = TimeUnit.SECONDS.convert(tokenRefreshDuration, TimeUnit.DAYS);//token允许刷新时间/分钟
        DecodedJWT jwtModel = null;
        try {
        	JWTVerifier verifier = JWT.require(Algorithm.HMAC256(SECRET)).build();
        	jwtModel = verifier.verify(oldToken);
        } catch (TokenExpiredException e) {
            try {
            	long issuedAt = jwtModel.getIssuedAt().toInstant().getEpochSecond();
            	long expiresAt = jwtModel.getExpiresAt().toInstant().getEpochSecond();
            	long nowTime = Instant.now().getEpochSecond();
            	long tokenTimeout = nowTime-expiresAt-issuedAt;
            	
                /*2.大于TIME_OUT 小于FORBID_REFRES_HTIME需要刷新*/
                if (tokenTimeout >= tokenDurationTime && tokenTimeout <= tokenRefreshDurationTime) {
                	Long point = jwtModel.getClaim(pointKey).asLong();
                    return createToken(point);
                }
            } catch (Exception ex) {
                throw new RuntimeException("会话刷新异常...", ex);
            }
        }
        /*3.超过FORBID_REFRES_HTIME 直接返回禁用刷新*/
        throw new RuntimeException("会话不允许刷新...");
    }
}
