package com.gogo.common.ui;

import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.List;

@SuppressWarnings({ "serial" })
@ApiModel
public class PageApi<T> implements Serializable {
	@ApiModelProperty("响应数据")
	private List<T> records;
	@ApiModelProperty("响应错误时，显示错误信息")
	private String[] msg;
	
	/**
	 * 1打头代表系统信息
	 *	1000：请求成功
	 * 
	 * 2打头代表业务信息
	 * 	2001：用户名或密码错误
	 */
	@ApiModelProperty("响应代码")
	private int code;
	
	private Boolean isSuccess;
	
	@ApiModelProperty("总记录数")
	private long allRow; // 总记录数
	@ApiModelProperty("总页数")
	private int totalPage; // 总页数
	@ApiModelProperty("当前页")
	private int currentPage; // 当前页
	@ApiModelProperty("每页记录数")
	private int pageSize; // 每页记录数
	private int count;

	@ApiModelProperty("是否第一页")
	private Boolean isFirstPage; // 是否为第一页
	@ApiModelProperty("是否最后一页")
	private Boolean isLastPage; // 是否为最后一页
	@ApiModelProperty("是否有前一页")
	private Boolean hasPreviousPage; // 是否有前一页
	@ApiModelProperty("是否有下一页")
	private Boolean hasNextPage; // 是否有下一页

	@ApiModelProperty("排序字段")
	private String orderBy;
	
	private int beginPage;
	private int pageLength;
	
	private static String[] defaultMsg = new String[] {"未知错误"};
	private static int defaultCode = 1000;
	private static int defaultErrorCode = 2000;
	
	private PageApi() {}
	
	public PageApi(List<T> records, long allRow, int currentPage, int pageSize) {
		this.records = records;
		this.allRow = allRow;
		this.currentPage = currentPage;
		this.pageSize = pageSize;
		this.count = records.size();
		this.code = defaultCode;
		this.isSuccess = true;
		init();
	}
	
	public PageApi(List<T> data, long allRow, int currentPage, int pageSize, String orderBy) {
		this(data, allRow, currentPage, pageSize);
		if(StringUtils.isNotBlank(orderBy)){
			this.orderBy = orderBy;
		}
	}
	
	public static <T> PageApi<T> success() {
		return success(null, 0l, new PageParam());
	}

	public static <T> PageApi<T> success(List<T> data, long allRow, PageParam page) {
		int currentPage = 1;
		int pageSize = Integer.MAX_VALUE-1;
		String orderBy = null;
		if(page != null) {
			currentPage = page.getPageNum();
			pageSize = page.getPageSize();
			orderBy = page.getOrderBy();
		}
		PageApi<T> pageApi = new PageApi<T>(data, allRow, currentPage, pageSize, orderBy);
		return pageApi;
	}
	
	public static <T> PageApi<T> error(String msg) {
		String[] msgs = new String[]{msg};
		return error(defaultErrorCode, msgs);
	}
	
	public static <T> PageApi<T> error(String[] msgs) {
		return error(defaultErrorCode, msgs);
	}
	
	public static <T> PageApi<T> error(int code, String msg) {
		String[] msgs = new String[]{msg};
		return error(defaultErrorCode, msgs);
	}
	
	public static <T> PageApi<T> error(int code, String[] msgs) {
		PageApi<T> pageApi = new PageApi<T>();
		pageApi.setCode(code);
		pageApi.setMsg(msgs);
		pageApi.setIsSuccess(false);
		return pageApi;
	}

	public void init() {
		// 计算当前页
		countCurrentPage();
		// 计算总页数
		countTotalPage();
		this.isFirstPage = isFirstPage();
		this.isLastPage = isLastPage();
		this.hasPreviousPage = isHasPreviousPage();
		this.hasNextPage = isHasNextPage();
		// p:9 每次显示9 个页码
		// t:20 总数
		// c:2当前页
		// b:1 开始页

		// c:18 12,13,14,15,16,17,18,19,20
		// b:20-9

		// c:5
		// b:0

		// c:6
		// b:1

		// c:16
		// b: 11,12,13
		if (this.currentPage - 4 <= 0) {
			this.beginPage = 1;
		} else if (this.currentPage + 4 >= this.totalPage) {
			this.beginPage = this.totalPage - 8;// 17 20-8=12
												// 开始页12，那么12,13，14,15,16,17,18,19,20
			if (beginPage <= 0) {
				this.beginPage = 1;
			}
		} else {
			// 18-5 13 13,14,15,16,17,18
			this.beginPage = currentPage - 4;
		}
		this.pageLength = this.totalPage - this.beginPage; // 20-12=8
		// pagingResult.beginPage+pagingResult.pageLength end pageLength --->end
		if (this.pageLength >= 8) {
			this.pageLength = 8;
		} else if (this.pageLength < 0) {
			this.pageLength = 0;
		}
	}

	public boolean isFirstPage() {
		return currentPage == 1; // 如是当前页是第1页
	}

	public boolean isLastPage() {
		return currentPage == totalPage; // 如果当前页是最后一页
	}

	public boolean isHasPreviousPage() {
		return currentPage != 1; // 只要当前页不是第1页
	}

	public boolean isHasNextPage() {
		return currentPage != totalPage; // 只要当前页不是最后1页
	}
	public void countTotalPage() {
		totalPage = (int) (allRow % pageSize == 0 ? allRow / pageSize : allRow / pageSize + 1);
	}
	public void countCurrentPage() {
		currentPage = (currentPage == 0 ? 1 : currentPage);
	}

	public List<T> getRecords() {
		return records;
	}

	public void setRecords(List<T> records) {
		this.records = records;
	}

	public String[] getMsg() {
		return msg;
	}
	public void setMsg(String[] msg) {
		this.msg = msg;
	}
	public int getCode() {
		return code;
	}
	public void setCode(int code) {
		this.code = code;
	}
	public long getAllRow() {
		return allRow;
	}
	public void setAllRow(long allRow) {
		this.allRow = allRow;
	}
	public int getTotalPage() {
		return totalPage;
	}
	public void setTotalPage(int totalPage) {
		this.totalPage = totalPage;
	}
	public int getCurrentPage() {
		return currentPage;
	}
	public void setCurrentPage(int currentPage) {
		this.currentPage = currentPage;
	}
	public int getPageSize() {
		return pageSize;
	}
	public void setPageSize(int pageSize) {
		this.pageSize = pageSize;
	}
	public String getOrderBy() {
		return orderBy;
	}
	public void setOrderBy(String orderBy) {
		this.orderBy = orderBy;
	}
	public int getBeginPage() {
		return beginPage;
	}
	public void setBeginPage(int beginPage) {
		this.beginPage = beginPage;
	}
	public int getPageLength() {
		return pageLength;
	}
	public void setPageLength(int pageLength) {
		this.pageLength = pageLength;
	}

	public Boolean isSuccess() {
		return isSuccess;
	}

	public void setIsSuccess(Boolean isSuccess) {
		this.isSuccess = isSuccess;
	}

	public int getCount() {
		return count;
	}

	public void setCount(int count) {
		this.count = count;
	}
	
}
