package com.me.art.widget.recycler;

import android.view.View;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

import com.me.art.utils.ThirdViewUtil;

/**
 * 基类 {@link RecyclerView.ViewHolder}
 *
 * @Author: lhy
 * @CreateDate: 3/12/21
 * @Description:
 */
public abstract class BaseHolder<T> extends RecyclerView.ViewHolder implements View.OnClickListener {
    protected final String TAG = this.getClass().getSimpleName();
    protected OnViewClickListener mOnViewClickListener;


    public BaseHolder(@NonNull View itemView) {
        super(itemView);
        itemView.setOnClickListener(this);
        //绑定ButterKnife
        ThirdViewUtil.bindTarget(this, itemView);
    }

    /**
     * 设置数据
     *
     * @param data     数据
     * @param position 在 RecyclerView 中的位置
     */
    public abstract void setData(@NonNull T data, int position);

    /**
     * 在 Activity 的 onDestroy 中使用 {@link DefaultAdapter#releaseAllHolder(RecyclerView)} 方法 (super.onDestroy() 之前)
     * {@link BaseHolder#onRelease()} 才会被调用, 可以在此方法中释放一些资源
     */
    protected void onRelease() {

    }

    @Override
    public void onClick(View v) {
        if (mOnViewClickListener != null)
            mOnViewClickListener.onViewClick(v, getAdapterPosition());
    }

    /**
     * item 点击事件
     */
    public interface OnViewClickListener {
        /**
         * item 被点击
         *
         * @param view     被点击的 {@link View}
         * @param position 在 RecyclerView 中的位置
         */
        void onViewClick(View view, int position);
    }

    public void setOnItemClickListener(OnViewClickListener listener) {
        this.mOnViewClickListener = listener;
    }
}
