package com.me.art.mvp;

import android.app.Activity;
import android.app.Service;
import android.view.View;

import androidx.fragment.app.Fragment;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.LifecycleObserver;
import androidx.lifecycle.LifecycleOwner;
import androidx.lifecycle.OnLifecycleEvent;

import com.me.art.integration.EventBusManager;
import com.me.art.utils.Preconditions;

import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.disposables.Disposable;


/**
 * @Author: lihanyang
 * @CreateDate: 11/30/20
 * @Description:
 */
public class BasePresenter<M extends IModel> implements IPresenter, LifecycleObserver {
    protected final String TAG = this.getClass().getSimpleName();
    protected CompositeDisposable mCompositeDisposable;
    protected M mModel;

    public BasePresenter() {
        onStart();
    }

    public BasePresenter(M model) {
        Preconditions.checkNotNull(model, "%s cannot be null", IModel.class.getName());
        this.mModel = model;
        onStart();
    }

    @Override
    public void onStart() {
        //如果使用EventBus 情将此方法返回true
        if (useEventBus())
            EventBusManager.getInstance().register(this);//注册 EventBus

    }

    @Override
    public void onDestroy() {
        if (useEventBus())
            //解除注册EventBus
            EventBusManager.getInstance().unregister(this);//注销 EventBus
        unDispose(); //解除订阅
        if (mModel != null)
            mModel.onDestroy();
        this.mModel = null;
        this.mCompositeDisposable = null;
    }

    /**
     * 只有当 {@code view} 实现了 {@link LifecycleOwner} 时, 此方法才会被调用
     * 所以当您想在 {@link Service} 以及一些自定义 {@link View} 或自定义类中使用 {@code Presenter} 时
     * 您也将不能继续使用 {@link OnLifecycleEvent} 绑定生命周期
     *
     * @param owner link {@link SupportActivity} and {@link Fragment}
     */
    @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
    void onDestroy(LifecycleOwner owner) {
        /**
         * 注意, 如果在这里调用了 {@link #onDestroy()} 方法, 会出现某些地方引用 {@code mModel} 为 null 的情况
         * 比如如果你声明了多个 @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY) 时在其他 @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
         * 中引用了 {@code mModel} 也可能会出现此情况
         */
        owner.getLifecycle().removeObserver(this);
    }

    /**
     * 是否使用EventBus,默认使用(true)
     *
     * @return
     */
    protected boolean useEventBus() {
        return true;
    }

    /**
     * 将 {@link Disposable} 添加到 {@link CompositeDisposable} 中统一管理
     * 可在 {@link Activity#onDestroy()} 中使用 {@link #unDispose()} 停止正在执行的 RxJava 任务,避免内存泄漏(框架已自行处理)
     *
     * @param disposable
     */
    protected void addDispose(Disposable disposable) {
        if (mCompositeDisposable == null)
            mCompositeDisposable = new CompositeDisposable();
        //将所有 Disposable 放入，集中处理
        mCompositeDisposable.add(disposable);
    }

    /**
     * 停止集合中正在执行的 RxJava 任务
     */
    public void unDispose() {
        if (mCompositeDisposable != null)
            mCompositeDisposable.clear();//保证 Activity 结束时取消所有正在执行的订阅
    }
}
