package com.hx.lib_common.utils;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.speech.tts.TextToSpeech;
import android.speech.tts.UtteranceProgressListener;
import android.util.Log;
import android.widget.Toast;

import com.hx.lib_common.log.HLog;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/4/2 16:50
 * Describe:
 */
public class TtsUtils {
    private static final String TAG = "TtsUtils";
    private TextToSpeech mTextToSpeech;
    private Context mContext;
    private Handler mHandler;
    private TtsPlayListener mListener;
    private String mSpeakMessage;

    private static final class SingletonHolder {
        private static final TtsUtils INSTANCE = new TtsUtils();
    }

    public static TtsUtils getInstance() {
        return SingletonHolder.INSTANCE;
    }

    private TtsUtils() {

    }

    /**
     * 初始化
     */
    public void init(Context context) {
        mContext = context.getApplicationContext();
        mTextToSpeech = new TextToSpeech(mContext, new TextToSpeech.OnInitListener() {
            @Override
            public void onInit(int status) {
                if (status == TextToSpeech.SUCCESS) {
                    int result = mTextToSpeech.setLanguage(Locale.CHINA);
                    if (result == TextToSpeech.LANG_MISSING_DATA || result == TextToSpeech.LANG_NOT_SUPPORTED) {
                        Toast.makeText(mContext, "语音数据丢失或不支持", Toast.LENGTH_LONG).show();
                    }
                }
            }
        });
    }

    /**
     * 朗读
     * @param message
     */
    public void speak(final String message){
        speak(message, null);
    }

    /**
     * 朗读
     * @param message
     */
    public void speak(final String message, final TtsPlayListener listener) {
        if (mHandler == null) {
            mHandler = new Handler(Looper.getMainLooper());
        }
        if (mTextToSpeech != null && !mTextToSpeech.isSpeaking()) {
            mTextToSpeech.setPitch(1.2f);// 设置音调，值越大声音大
            HashMap<String, String> ttsOption = new HashMap<>();
            ttsOption.put(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID, message);
            mTextToSpeech.setOnUtteranceProgressListener(new UtteranceProgressListener() {
                @Override
                public void onStart(String utteranceId) {
                    Log.d(TAG, "onStart() called with: utteranceId = [" + utteranceId + "]");
                    mHandler.removeCallbacks(checkTtsRunnable);
                }

                @Override
                public void onDone(String utteranceId) {
                    Log.d(TAG, "onDone() called with: utteranceId = [" + utteranceId + "]");
                    if (listener != null) {
                        listener.onSuccess(utteranceId);
                    }
                }

                @Override
                public void onError(String utteranceId) {
                    HLog.e(TAG, "speak onError " + utteranceId);
                    if (listener != null) {
                        listener.onFailed(utteranceId);
                    }
                }
            });
            mSpeakMessage = message;
            mListener = listener;
            mHandler.postDelayed(checkTtsRunnable, 1000);
            mTextToSpeech.speak(message, TextToSpeech.QUEUE_FLUSH, ttsOption);
        } else {
            HLog.w(TAG, "speak in speak: " + message);
            if (listener != null) {
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        listener.onFailed(message);
                    }
                });
            }
        }
    }

    /**
     * 停止朗读
     */
    public void stopSpeak() {
        if (mTextToSpeech != null) {
            mTextToSpeech.stop();//停止朗读
        }
    }

    /**
     * 释放资源
     */
    public void release() {
        if (mTextToSpeech != null) {
            mTextToSpeech.shutdown();
        }
    }

    private Runnable checkTtsRunnable = new Runnable() {
        @Override
        public void run() {
            HLog.e(TAG, "没有开始播报:" + mSpeakMessage);
//            init(mContext);
            if (mListener != null) {
                mListener.onFailed(mSpeakMessage);
            }
        }
    };

    public interface TtsPlayListener {
        void onSuccess(String message);

        void onFailed(String message);
    }
}
