package com.hx.lib_common.utils;

import com.hx.lib_common.constant.MemoryConst;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/4/8 17:19
 * Describe:
 * 流工具
 * bytes --> ByteArrayInputStream --> inputStream
 *
 * String --> bytes --> ByteArrayOutputStream
 * OutputStream --> ByteArrayOutputStream --> bytes
 */
public class StreamUtils {

    private static final int BUFFER_SIZE = 8* MemoryConst.KB;

    /**
     * inputStream--》bytes
     * @param is
     * @return
     */
    public static byte[] inputStream2Bytes(final InputStream is) {
        if (is == null) return null;
        return inputStream2OutputStream(is).toByteArray();
    }

    /**
     * inputStream--》String
     * @param is
     * @param charsetName
     * @return
     */
    public static String inputStream2String(final InputStream is, final String charsetName) {
        if (is == null) return "";
        try {
            ByteArrayOutputStream baos = inputStream2OutputStream(is);
            if (baos == null) return "";
            return baos.toString(getSafeCharset(charsetName));
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return "";
        }
    }

    /**
     * inputStream--》List String(UTF-8)
     * @param is
     * @return
     */
    public static List<String> inputStream2Lines(final InputStream is) {
        return inputStream2Lines(is, "");
    }

    /**
     * inputStream--》List String
     * @param is
     * @param charsetName
     * @return
     */
    public static List<String> inputStream2Lines(final InputStream is, final String charsetName) {
        BufferedReader reader = null;
        InputStreamReader inputReader = null;
        try {
            List<String> list = new ArrayList<>();
            inputReader = new InputStreamReader(is, getSafeCharset(charsetName));
            reader = new BufferedReader(inputReader);
            String line;
            while ((line = reader.readLine()) != null) {
                list.add(line);
            }
            return list;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            CloseUtils.closeIO(inputReader, reader);
        }
    }

    /**
     * inputStream--》OutputStream
     * @param is
     * @return
     */
    public static ByteArrayOutputStream inputStream2OutputStream(final InputStream is) {
        if (is == null) return null;
        try {
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            byte[] b = new byte[BUFFER_SIZE];
            int len;
            while ((len = is.read(b, 0, BUFFER_SIZE)) != -1) {
                os.write(b, 0, len);
            }
            return os;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            CloseUtils.closeIO(is);
        }
    }

    /**
     * bytes--》inputStream
     * @param bytes
     * @return
     */
    public static InputStream bytes2InputStream(final byte[] bytes) {
        if (bytes == null || bytes.length <= 0) return null;
        return new ByteArrayInputStream(bytes);
    }

    /**
     * String--》inputStream
     * @param string
     * @param charsetName
     * @return
     */
    public static InputStream string2InputStream(final String string, final String charsetName) {
        if (string == null) return null;
        try {
            return new ByteArrayInputStream(string.getBytes(getSafeCharset(charsetName)));
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * outputStream--》inputStream
     * @param out
     * @return
     */
    public ByteArrayInputStream outputStream2InputStream(final OutputStream out) {
        if (out == null) return null;
        return new ByteArrayInputStream(outputStream2Bytes(out));
    }

    /**
     * outputStream-->Bytes
     * @param out
     * @return
     */
    public static byte[] outputStream2Bytes(final OutputStream out) {
        if (out == null) return null;
        return ((ByteArrayOutputStream) out).toByteArray();
    }

    /**
     * Bytes-->outputStream
     */
    public static OutputStream bytes2OutputStream(final byte[] bytes) {
        if (bytes == null || bytes.length <= 0) return null;
        ByteArrayOutputStream os = null;
        try {
            os = new ByteArrayOutputStream();
            os.write(bytes);
            return os;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            CloseUtils.closeIO(os);
        }
    }

    /**
     * String-->outputStream
     * @param string
     * @param charsetName
     * @return
     */
    public static OutputStream string2OutputStream(final String string, final String charsetName) {
        if (string == null) return null;
        try {
            return bytes2OutputStream(string.getBytes(getSafeCharset(charsetName)));
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return null;
        }
    }

    private static String getSafeCharset(String charsetName) {
        String cn = charsetName;
        if (StringUtils.isSpace(charsetName) || !Charset.isSupported(charsetName)) {
            cn = "UTF-8";
        }
        return cn;
    }

}
