package com.hx.lib_common.utils;

import android.content.Context;
import android.net.TrafficStats;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Process;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Enumeration;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/5/15 9:33
 * Describe:
 */
public class NetworkUtils {

    /**
     * 是否打开wifi
     *
     * @param context
     * @return
     */
    public static boolean isWifiEnabled(Context context) {
        WifiManager wifiManager = (WifiManager) context.getSystemService(Context.WIFI_SERVICE);
        return wifiManager.isWifiEnabled();
    }

    /**
     * 获取wifi的ip
     *
     * @param context
     * @return
     */
    public static String getWifiIp(Context context) {
        //获取wifi服务
        WifiManager wifiManager = (WifiManager) context.getSystemService(Context.WIFI_SERVICE);
        WifiInfo wifiInfo = wifiManager.getConnectionInfo();
        int ipAddress = wifiInfo.getIpAddress();
        if (ipAddress != 0) {
            return intToIp(ipAddress);
        }
        return null;
    }

    /**
     * 获取本地ip
     *
     * @return
     */
    public static String getLocalIp() {
        try {
            for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements(); ) {
                NetworkInterface intf = en.nextElement();
                for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements(); ) {
                    InetAddress inetAddress = enumIpAddr.nextElement();
                    if (!inetAddress.isLoopbackAddress()) {
                        String address = inetAddress.getHostAddress();
                        if (address.indexOf(".") > 0) {
                            return address;
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取adb调试端口
     *
     * @return
     */
    public static String getAdbDebugPort() {
        try {
            return (String) new Reflect("android.os.SystemProperties")
                    .invokeMethod("get", "service.adb.tcp.port");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 设置adb调试端口
     *
     * @param port
     * @return
     */
    public static void setAdbDebugPort(int port) {
        ShellUtils.execCommand(new String[]{String.format("setprop service.adb.tcp.port %d", port),
                "stop adbd", "start adbd"
        }, true);
    }

    private static String intToIp(int i) {
        return (i & 0xFF) + "." +
                ((i >> 8) & 0xFF) + "." +
                ((i >> 16) & 0xFF) + "." +
                (i >> 24 & 0xFF);
    }


    /**
     * 获取手机卡流量信息
     *
     * @return
     */
    public static @Nullable FlowInfo getMobileFlowInfo() {
        return getFlowInfo("rmnet0:");
    }

    /**
     * 获取以太网流量
     * @return
     */
    public static @Nullable FlowInfo getEthernetFlowInfo(){
        return getFlowInfo("eth0:");
    }

    /**
     * 获取wifi流量
     * @return
     */
    public static @Nullable FlowInfo getWifiFlowInfo(){
        return getFlowInfo("wlan0:");
    }

    /**
     * 获取用户流量
     * @return
     */
    public static @Nullable FlowInfo getUidFlowInfo(){
        int uid = Process.myUid();
        long rxBytes = TrafficStats.getUidRxBytes(uid);
        long txBytes = TrafficStats.getUidTxBytes(uid);
        return new FlowInfo(rxBytes == TrafficStats.UNSUPPORTED? 0: rxBytes, txBytes == TrafficStats.UNSUPPORTED?0: rxBytes);
    }


    /**
     * 获取流量信息(总量)
     * @param name 可以是wlan0:(wifi)/rmnet0:(手机流量)/eth0:(以太网)
     * @return
     */
    private static @Nullable FlowInfo getFlowInfo(@NonNull String name) {
        try {
            String content = FileIoUtils.readFile2String("/proc/net/dev");
            if (TextUtils.isEmpty(content)) {
                return null;
            }
            int startIndex = content.indexOf(name);
            if (startIndex < 0) {
                return null;
            }
            content = content.substring(startIndex);
            Pattern p = Pattern.compile(" \\d+");
            Matcher matcher = p.matcher(content);
            int count = 0;
            long uploadByte = 0;
            long downloadByte = 0;
            while (matcher.find()) {
                String group = matcher.group();
                if (count == 0) {
                    downloadByte = Long.parseLong(group.trim());
                }
                if (count == 8) {
                    uploadByte = Long.parseLong(group.trim());
                    return new FlowInfo(downloadByte, uploadByte);
                }
                count++;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 流量信息
     */
    public static class FlowInfo {
        /**
         * 下载流量
         */
        private long downloadBytes;
        /**
         * 上传流量
         */
        private long uploadBytes;

        public FlowInfo(long downloadBytes, long uploadBytes) {
            this.downloadBytes = downloadBytes;
            this.uploadBytes = uploadBytes;
        }

        public long getDownloadBytes() {
            return downloadBytes;
        }

        public long getUploadBytes() {
            return uploadBytes;
        }

        @Override
        public String toString() {
            return "FlowInfo{" +
                    "downloadBytes=" + downloadBytes +
                    ", uploadBytes=" + uploadBytes +
                    '}';
        }
    }

}
