package com.hx.lib_common.utils;

import android.annotation.TargetApi;
import android.app.ActivityManager;
import android.app.Application;
import android.content.Context;
import android.os.Build;
import android.os.Debug;
import android.os.Environment;
import android.os.Process;
import android.os.StatFs;
import android.os.SystemClock;
import android.provider.Settings;
import android.text.TextUtils;
import android.text.format.Formatter;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.WindowManager;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/4/9 17:12
 * Describe:
 * 设备相关
 */
public class DeviceUtils {

    /**
     * 获取设备序列号
     */
    public static String getSerialNumber() {
        String serial = null;
        try {
            Reflect reflect = new Reflect("android.os.SystemProperties");
            serial = (String) reflect.invokeMethod("get", "ro.boot.serialno");
            if (TextUtils.isEmpty(serial)) {
                serial = (String) reflect.invokeMethod("get", "ro.serialno");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return serial;
    }

    /**
     * 获取屏幕的宽度（单位：px）
     * @param context 上下文
     * @return 屏幕宽px
     */
    public static int getScreenWidth(Context context) {
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        DisplayMetrics dm = new DisplayMetrics();// 创建了一张白纸
        windowManager.getDefaultDisplay().getMetrics(dm);// 给白纸设置宽高
        return dm.widthPixels;
    }

    /**
     * 获取屏幕的高度（单位：px）
     * @param context 上下文
     * @return 屏幕高px
     */
    public static int getScreenHeight(Context context) {
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        DisplayMetrics dm = new DisplayMetrics();// 创建了一张白纸
        windowManager.getDefaultDisplay().getMetrics(dm);// 给白纸设置宽高
        return dm.heightPixels;
    }

    /**
     * 获取屏幕的宽度（单位：px）
     * @param context 上下文
     * @return 屏幕宽px
     */
    public static int getRealScreenWidth(Context context) {
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        DisplayMetrics dm = new DisplayMetrics();// 创建了一张白纸
        windowManager.getDefaultDisplay().getRealMetrics(dm);// 给白纸设置宽高
        return dm.widthPixels;
    }

    /**
     * 获取屏幕的高度（单位：px）
     * @param context 上下文
     * @return 屏幕高px
     */
    public static int getRealScreenHeight(Context context) {
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        DisplayMetrics dm = new DisplayMetrics();// 创建了一张白纸
        windowManager.getDefaultDisplay().getRealMetrics(dm);// 给白纸设置宽高
        return dm.heightPixels;
    }

    /**
     * 获取屏幕DPI
     * @param context 上下文
     * @return 屏幕高px
     */
    public static int getScreenDensity(Context context) {
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        DisplayMetrics dm = new DisplayMetrics();// 创建了一张白纸
        windowManager.getDefaultDisplay().getMetrics(dm);// 给白纸设置宽高
        return dm.densityDpi;
    }

    /**
     * 导航栏高度
     * @param context
     * @return
     */
    public static int getNaviBarHeight(Context context){
        int resourceId = context.getResources().getIdentifier("navigation_bar_height", "dimen", "android");
        int naviBarHeight = 0;
        if (resourceId != 0) {
            naviBarHeight = context.getResources().getDimensionPixelSize(resourceId);
        }
        return naviBarHeight;
    }

    /**
     * 获取内存可用内存空间
     *
     * @return
     */
    public static String getAvailMemory() {
        ActivityManager am = (ActivityManager) AppUtils.getApplication().getSystemService(Context.ACTIVITY_SERVICE);
        ActivityManager.MemoryInfo mi = new ActivityManager.MemoryInfo();
        am.getMemoryInfo(mi);
        //mi.availMem; 当前系统的可用内存
        return Formatter.formatFileSize(AppUtils.getApplication(), mi.availMem);// 将获取的内存大小规格化
    }

    /**
     * 获取内存总共空间
     * @return
     */
    public static String getTotalMemory() {
        String str1 = "/proc/meminfo";// 系统内存信息文件
        String str2;
        String[] arrayOfString;
        long initial_memory = 0;
        try {
            FileReader localFileReader = new FileReader(str1);
            BufferedReader localBufferedReader = new BufferedReader(
                    localFileReader, 8192);
            str2 = localBufferedReader.readLine();// 读取meminfo第一行，系统总内存大小

            arrayOfString = str2.split("\\s+");
            initial_memory = Integer.valueOf(arrayOfString[1]) * 1024L;// 获得系统总内存，单位是KB，乘以1024转换为Byte
            CloseUtils.closeIO(localBufferedReader, localFileReader);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return Formatter.formatFileSize(AppUtils.getApplication(), initial_memory);// Byte转换为KB或者MB，内存大小规格化
    }

    /**
     * 获取App的内存信息，与AS的Profile是一样的
     * @return
     */
    @TargetApi(Build.VERSION_CODES.M)
    public static String getAppMemoryInfo(){
        Application application = AppUtils.getApplication();
        ActivityManager am = (ActivityManager) application.getSystemService(Context.ACTIVITY_SERVICE);
        Debug.MemoryInfo[] memoryInfo = am.getProcessMemoryInfo(new int[]{Process.myPid()});
        if (memoryInfo != null && memoryInfo.length > 0){
            String maxMemory = Formatter.formatFileSize(AppUtils.getApplication(), Runtime.getRuntime().maxMemory());
            String total = memoryInfo[0].getMemoryStat("summary.total-pss");
            String java = memoryInfo[0].getMemoryStat("summary.java-heap");
            String nativeSize = memoryInfo[0].getMemoryStat("summary.native-heap");
            String graphics = memoryInfo[0].getMemoryStat("summary.graphics");
            String stack = memoryInfo[0].getMemoryStat("summary.stack");
            String code = memoryInfo[0].getMemoryStat("summary.code");
            String system = memoryInfo[0].getMemoryStat("summary.system");
            return String.format("max:%s,total-pss:%s,java:%s,native:%s,graphics:%s,stack:%s,code:%s,system:%s", maxMemory,total, java, nativeSize, graphics, stack, code, system);
        }
        return null;
    }

    /**
     * 获取cpu使用率
     *
     * @return (user 、 nice 、 system 、 idle 、 iowait 、 irq 、 softirq 、 stealstealon 、 guest)
     * totalCpuTime = user + nice + system + idle + iowait + irq + softirq
     */
    public static float getCpuUsed() {
        try {
            RandomAccessFile reader = new RandomAccessFile("/proc/stat", "r");
            String load = reader.readLine();
            String[] toks = load.split(" ");
            long idle1 = Long.parseLong(toks[5]);
            long cpu1 = Long.parseLong(toks[2]) + Long.parseLong(toks[3]) + Long.parseLong(toks[4])
                    + Long.parseLong(toks[6]) + Long.parseLong(toks[7]) + Long.parseLong(toks[8]);
            try {
                Thread.sleep(300);
            } catch (Exception e) {
                e.printStackTrace();
            }
            reader.seek(0);
            load = reader.readLine();
            CloseUtils.closeIO(reader);
            toks = load.split(" ");
            long idle2 = Long.parseLong(toks[5]);
            long cpu2 = Long.parseLong(toks[2]) + Long.parseLong(toks[3]) + Long.parseLong(toks[4])
                    + Long.parseLong(toks[6]) + Long.parseLong(toks[7]) + Long.parseLong(toks[8]);
            return (float) (cpu2 - cpu1) / ((cpu2 + idle2) - (cpu1 + idle1));
        } catch (IOException ex) {
            ex.printStackTrace();
        }
        return 0;
    }

    /**
     * 获取cpu温度
     *
     * @return
     */
    public static String getCpuTemp() {
        String temperature = "Unknow";
        try {
            FileReader fr = new FileReader("/sys/class/thermal/thermal_zone0/type");
            BufferedReader br = new BufferedReader(fr);
            String type = br.readLine();
            fr = new FileReader("/sys/class/thermal/thermal_zone0/temp");
            br = new BufferedReader(fr);
            temperature = br.readLine();
            //Qualcomm高通
            if (type.contains("tsens_tz_sensor")) {
                // Qualcomm CPU
                long temp = Long.parseLong(temperature);
                if (temp < 0) {
                    temperature = "Unknow";
                } else if (temp > 100) {
                    temperature = (float) (temp / 10.0) + "";
                } else {
                    temperature = temp + "";
                }
            }
            //MTK或其它
//        if (type.contains("cpu")) {
            else {
                long temp = Long.parseLong(temperature);
                if (temp < 0) {
                    temperature = "Unknow";
                } else {
                    temperature = (float) (temp / 1000.0) + "";
                }
            }
            CloseUtils.closeIO(fr, br);
        } catch (IOException e) {
        }
        return temperature;
    }

    /**
     * 外部存储是否可用 (存在且具有读写权限)
     */
    public static boolean isExternalStorageAvailable() {
        return Environment.getExternalStorageState().equals(
                Environment.MEDIA_MOUNTED);
    }

    /**
     * 获取手机外部可用空间大小
     * @return
     */
    public static String getAvailableExternalStorage() {
        if (isExternalStorageAvailable()) {
            StatFs stat = new StatFs(Environment.getExternalStorageDirectory().getAbsolutePath());
            long blockSize = stat.getBlockSizeLong();
            long availableBlocks = stat.getAvailableBlocksLong();
            return Formatter.formatFileSize(AppUtils.getApplication(), availableBlocks * blockSize);
        } else {
            return "-1";
        }
    }

    /**
     * 获取手机外部总空间大小
     * @return
     */
    public static String getTotalExternalStorage() {
        if (isExternalStorageAvailable()) {
            StatFs stat = new StatFs(Environment.getExternalStorageDirectory().getAbsolutePath());
            long blockSize = stat.getBlockSizeLong();
            long totalBlocks = stat.getBlockCountLong();
            return Formatter.formatFileSize(AppUtils.getApplication(), totalBlocks * blockSize);
        } else {
            return "-1";
        }
    }

    /**
     * 获取Kernel版本
     * @return
     */
    public static String getKernelVersion() {
        String path = "/proc/version";
        String version = "";
        try {
            FileReader localFileReader = new FileReader(path);
            BufferedReader localBufferedReader = new BufferedReader(
                    localFileReader, 8192);
            version = localBufferedReader.readLine();
            CloseUtils.closeIO(localBufferedReader, localFileReader);
        } catch (IOException e) {
        }
        return version;
    }

    /**
     * 获取系统版本（打包版本）
     *
     * @return
     */
    public static String getDisplayVersion() {
        return Build.DISPLAY;
    }

    /**
     * 获取Android版本，如7.1.2
     *
     * @return
     */
    public static String getAndroidVersion() {
        return Build.VERSION.RELEASE;
    }

    /**
     * 获取设备类型
     *
     * @return
     */
    public static String getModel() {
        return Build.MODEL;
    }

    /**
     * 获取制造厂家
     *
     * @return
     */
    public static String getManufacture() {
        return Build.MANUFACTURER;
    }

    /**
     * 获取以太网MAC地址
     *
     * @return
     */
    public static String getEthernetMacAddress() {
        String path = "/sys/class/net/eth0/address";
        String result = "";
        try {
            FileReader localFileReader = new FileReader(path);
            BufferedReader localBufferedReader = new BufferedReader(
                    localFileReader, 8192);
            result = localBufferedReader.readLine();
            CloseUtils.closeIO(localBufferedReader, localFileReader);
        } catch (IOException e) {
        }
        return result;
    }

}
