package com.hx.lib_common.utils;

import androidx.annotation.IntDef;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/4/8 9:44
 * Describe:
 */
public class ThreadPoolUtils {

    private static final int   CPU_COUNT = Runtime.getRuntime().availableProcessors();

    /**
     * 固定个数，超出的线程会在队列中等待。
     * 大小可根据 Runtime.getRuntime().availableProcessors()
     */
    public static final int TYPE_FIXED = 0;
    /**
     * 可缓存线程池，线程池为无限大，
     * 当执行第二个任务时第一个任务已经完成，会复用执行第一个任务的线程，而不用每次新建线程
     */
    public static final int TYPE_CACHED = 1;
    /**
     * 单一线程
     * 所有任务按照指定顺序(FIFO, LIFO, 优先级)执行
     */
    public static final int TYPE_SINGLE = 2;
    /**
     * 大小无限制的线程池，支持定时和周期性的执行线程
     */
    public static final int TYPE_SCHEDULED = 3;

    @IntDef({TYPE_FIXED, TYPE_CACHED, TYPE_SINGLE, TYPE_SCHEDULED})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Type {
    }

    private ExecutorService exec;

    public static ThreadPoolUtils getFixedPool(int corePoolSize){
        return new ThreadPoolUtils(TYPE_FIXED, corePoolSize);
    }

    public static ThreadPoolUtils getCachedPool(){
        return new ThreadPoolUtils(TYPE_CACHED, 0);
    }

    public static ThreadPoolUtils getSinglePool(){
        return new ThreadPoolUtils(TYPE_SINGLE, 0);
    }

    public static ThreadPoolUtils getScheduledPool(){
        return new ThreadPoolUtils(TYPE_SCHEDULED, CPU_COUNT);
    }

    /**
     * ThreadPoolUtils构造函数
     *
     * @param type 线程池类型
     * @param corePoolSize 只对Fixed和Scheduled线程池起效
     */
    public ThreadPoolUtils(@Type int type, int corePoolSize) {
        switch (type) {
            case TYPE_FIXED:
                exec = Executors.newFixedThreadPool(corePoolSize);
                break;
            case TYPE_SINGLE:
                exec = Executors.newSingleThreadExecutor();
                break;
            case TYPE_CACHED:
                exec = Executors.newCachedThreadPool();
                break;
            case TYPE_SCHEDULED:
                exec = Executors.newScheduledThreadPool(corePoolSize);
                break;
        }
    }

    /**
     * 在未来某个时间执行给定的命令
     * 该命令可能在新的线程、已入池的线程或者正调用的线程中执行，这由 Executor 实现决定。
     *
     * @param command
     */
    public void execute(Runnable command) {
        exec.execute(command);
    }

    /**
     * 在未来某个时间执行给定的命令链表
     * 该命令可能在新的线程、已入池的线程或者正调用的线程中执行，这由 Executor 实现决定。
     * @param commands 命令链表
     */
    public void execute(List<Runnable> commands) {
        for (Runnable command : commands) {
            exec.execute(command);
        }
    }

    /**
     * 待以前提交的任务执行完毕后关闭线程池
     * 启动一次顺序关闭，执行以前提交的任务，但不接受新任务。 如果已经关闭，则调用没有作用。
     */
    public void shutDown() {
        exec.shutdown();
    }

    /**
     * 试图停止所有正在执行的活动任务
     * 试图停止所有正在执行的活动任务，暂停处理正在等待的任务，并返回等待执行的任务列表。
     * 无法保证能够停止正在处理的活动执行任务，但是会尽力尝试。
     * @return 等待执行的任务的列表
     */
    public List<Runnable> shutDownNow() {
        return exec.shutdownNow();
    }

    /**
     * 判断线程池是否已关闭
     */
    public boolean isShutDown() {
        return exec.isShutdown();
    }

    /**
     * 关闭线程池后判断所有任务是否都已完成
     * 注意，除非首先调用 shutdown 或 shutdownNow，否则 isTerminated 永不为 true。
     */
    public boolean isTerminated() {
        return exec.isTerminated();
    }

}
