package com.hx.lib_common.utils;

import android.annotation.SuppressLint;

import java.util.concurrent.TimeUnit;

import io.reactivex.BackpressureStrategy;
import io.reactivex.Flowable;
import io.reactivex.FlowableEmitter;
import io.reactivex.FlowableOnSubscribe;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Action;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Function;
import io.reactivex.observers.DisposableObserver;
import io.reactivex.schedulers.Schedulers;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/3/27 15:39
 * Describe:
 * RxJava实现的任务调试工具类
 */
public class TaskUtils {
    private static final String TAG = "TaskUtils";

    /**
     * 倒计时
     *
     * @param total
     * @param task
     * @return
     */
    public static TaskObject countDown(final long total, final CountDownTask task) {
        return new TaskObject(Flowable.interval(0, 1, TimeUnit.SECONDS)
                .onBackpressureLatest()
                .take(total)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .map(new Function<Long, Long>() {
                    @Override
                    public Long apply(Long count) throws Exception {
                        return total - count - 1;
                    }
                })
                .doOnNext(new Consumer<Long>() {
                    @Override
                    public void accept(Long count) throws Exception {
                        task.onCounting(count);
                    }
                })
                .doOnComplete(new Action() {
                    @Override
                    public void run() throws Exception {
                        task.onFinish();
                    }
                })
                .subscribe());

    }

    /**
     * 循环调度（主线程）
     *
     * @param initialDelay
     * @param period
     * @param task
     * @return
     */
    public static TaskObject intervalTask(long initialDelay, long period, Task task) {
        DisposableObserver<Long> observer = createObserver(task);
        Observable.interval(initialDelay, period, TimeUnit.MILLISECONDS)
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(observer);
        return new TaskObject(observer);
    }

    /**
     * 循环调度（非主线程）
     *
     * @param initialDelay
     * @param period
     * @param task
     * @return
     */
    public static TaskObject inervalAsyncTask(long initialDelay, long period, Task task) {
        DisposableObserver<Long> observer = createObserver(task);
        Observable.interval(initialDelay, period, TimeUnit.MILLISECONDS)
                .observeOn(Schedulers.io())
                .subscribe(observer);
        return new TaskObject(observer);
    }

    /**
     * 延时执行（主线程）
     *
     * @param delay
     * @param task
     * @return
     */
    public static TaskObject delayTask(long delay, final Task task) {
        DisposableObserver<Long> observer = createObserver(task);
        Observable.timer(delay, TimeUnit.MILLISECONDS)
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(observer);
        return new TaskObject(observer);
    }

    /**
     * 延时执行（非主线程）
     *
     * @param delay
     * @param task
     * @return
     */
    public static TaskObject delayAsyncTask(long delay, final Task task) {
        DisposableObserver<Long> observer = createObserver(task);
        Observable.timer(delay, TimeUnit.MILLISECONDS)
                .observeOn(Schedulers.io())
                .subscribe(observer);
        return new TaskObject(observer);
    }

    /**
     * 线程执行某个方法
     * @param task
     */
    public static void runAsync(final Task task) {
        Flowable.create(new FlowableOnSubscribe<Object>() {
            @Override
            public void subscribe(FlowableEmitter<Object> emitter) throws Exception {
                task.run();

            }
        }, BackpressureStrategy.BUFFER)
                .subscribeOn(Schedulers.io())
                .subscribe();


    }

    /**
     * 带返回值的执行某个方法
     * @param task
     * @param <T>
     */
    @SuppressLint("CheckResult")
    public static <T>void runAsyncWithResult(final ResultTask<T> task){
        Observable.create(new ObservableOnSubscribe<T>() {
            @Override
            public void subscribe(ObservableEmitter<T> emitter) throws Exception {
                emitter.onNext(task.run());
            }
        })
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<T>() {
                    @Override
                    public void accept(T result) throws Exception {
                        task.onResult(result);
                    }
                });
    }


    private static DisposableObserver<Long> createObserver(final Task task) {
        return new DisposableObserver<Long>() {
            @Override
            public void onNext(Long time) {
                task.run();
            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onComplete() {

            }
        };
    }

    public interface Task {
        void run();
    }

    public interface ResultTask<T> {
        T run();
        void onResult(T result);
    }

    public interface CountDownTask {
        void onFinish();

        void onCounting(Long count);
    }

    public static class TaskObject{
        private Disposable mDisposable;
        private TaskObject(Disposable disposable){
            mDisposable = disposable;
        }

        public void stopTask(){
            if (mDisposable != null && !mDisposable.isDisposed()){
                mDisposable.dispose();
                mDisposable = null;
            }
        }

        public boolean isRunning(){
            return mDisposable != null && !mDisposable.isDisposed();
        }
    }

}
