package com.hx.lib_common.utils;

import android.app.Application;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/4/2 14:08
 * Describe:
 */
public class Reflect {

    private final Class<?> clazz;
    private Object object;

    public Reflect(String className) throws ClassNotFoundException {
        clazz = forName(className);
    }

    public Reflect(String className, ClassLoader classLoader) throws ClassNotFoundException {
        clazz = forName(className, classLoader);
    }

    public Reflect newInstance(Object... args) throws NoSuchMethodException, IllegalAccessException, InvocationTargetException, InstantiationException {
        Class<?>[] argsType = getArgsType(args);
        try {
            Constructor<?> constructor = clazz.getDeclaredConstructor(argsType);
            constructor.setAccessible(true);
            object = constructor.newInstance(args);
            return this;
        }catch (NoSuchMethodException e) {
            for (Constructor<?> constructor : clazz.getDeclaredConstructors()) {
                if (match(constructor.getParameterTypes(), argsType)) {
                    constructor.setAccessible(true);
                    object = constructor.newInstance(args);
                    return this;
                }
            }
        }

        throw new NoSuchMethodException("No similar Constructor on type " + clazz + ".");
    }

    public Object invokeMethod(String methodName, Object... args) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
        Class<?>[] argsType = getArgsType(args);

        try {
            Method method = clazz.getDeclaredMethod(methodName, argsType);
            method.setAccessible(true);
            return method.invoke(object, args);
        } catch (NoSuchMethodException e) {
            Method method = similarMethod(methodName, argsType);
            method.setAccessible(true);
            return method.invoke(object, args);
        }
    }

    public Object getField(String fieldName) throws NoSuchFieldException, IllegalAccessException {
        try {
            Field field = field(fieldName);
            field.setAccessible(true);
            return field.get(object);
        } catch (NoSuchFieldException e) {
            Field field = similarField(fieldName);
            field.setAccessible(true);
            return field.get(object);
        }
    }

    public void setField(String fieldName, Object value) throws NoSuchFieldException, IllegalAccessException {
        try {
            Field field = field(fieldName);
            field.setAccessible(true);
            field.set(object, value);
        } catch (NoSuchFieldException e) {
            Field field = similarField(fieldName);
            field.setAccessible(true);
            field.set(object, value);
        }
    }

    private static Class<?> forName(String className) throws ClassNotFoundException {
        return Class.forName(className);
    }

    private static Class<?> forName(String name, ClassLoader classLoader) throws ClassNotFoundException {
        return Class.forName(name, true, classLoader);
    }

    private Class<?>[] getArgsType(final Object... args) {
        if (args == null) return new Class[0];
        Class<?>[] result = new Class[args.length];
        for (int i = 0; i < args.length; i++) {
            Object value = args[i];
            result[i] = value == null ? NULL.class : value.getClass();
        }
        return result;
    }

    private Method similarMethod(final String name, final Class<?>[] types)
            throws NoSuchMethodException {
        Class<?> cla = clazz;
        do {
            for (Method method : cla.getDeclaredMethods()) {
                if (isSimilarSignature(method, name, types)) {
                    return method;
                }
            }
            cla = cla.getSuperclass();
        } while (cla != null);

        throw new NoSuchMethodException("No similar method " + name + " with params "
                + Arrays.toString(types) + " could be found on type " + clazz + ".");
    }

    private Field similarField(String fieldName) throws NoSuchFieldException {
        Class<?> cla = clazz.getSuperclass();
        while (cla != null) {
            try {
                return cla.getDeclaredField(fieldName);
            } catch (NoSuchFieldException e) {
                cla = cla.getSuperclass();
            }
        }
        throw new NoSuchFieldException("No similar field " + fieldName + " on type " + clazz + ".");

    }

    private boolean isSimilarSignature(final Method possiblyMatchingMethod,
                                       final String desiredMethodName,
                                       final Class<?>[] desiredParamTypes) {
        return possiblyMatchingMethod.getName().equals(desiredMethodName)
                && match(possiblyMatchingMethod.getParameterTypes(), desiredParamTypes);
    }

    private boolean match(final Class<?>[] declaredTypes, final Class<?>[] actualTypes) {
        if (declaredTypes.length == actualTypes.length) {
            for (int i = 0; i < actualTypes.length; i++) {
                if (actualTypes[i] == NULL.class
                        || wrapper(declaredTypes[i]).isAssignableFrom(wrapper(actualTypes[i]))) {
                    continue;
                }
                return false;
            }
            return true;
        } else {
            return false;
        }
    }

    /**
     * 类型转换
     * @param type
     * @return
     */
    private Class<?> wrapper(final Class<?> type) {
        if (type == null) {
            return null;
        } else if (type.isPrimitive()) {
            if (boolean.class == type) {
                return Boolean.class;
            } else if (int.class == type) {
                return Integer.class;
            } else if (long.class == type) {
                return Long.class;
            } else if (short.class == type) {
                return Short.class;
            } else if (byte.class == type) {
                return Byte.class;
            } else if (double.class == type) {
                return Double.class;
            } else if (float.class == type) {
                return Float.class;
            } else if (char.class == type) {
                return Character.class;
            } else if (void.class == type) {
                return Void.class;
            }
        }
        return type;
    }

    private Field field(String fieldName) throws NoSuchFieldException {
        Field field = clazz.getDeclaredField(fieldName);
        field.setAccessible(true);
        return field;
    }

    private static class NULL {
    }
}
