package com.hx.lib_common.utils;

import android.os.Build;
import android.util.Log;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/4/9 15:26
 * Describe:
 * 时间工具类
 */
public class TimeUtils {
    private static final String TAG = "TimeUtils";
    /**
     * 时间格式
     * y	年	Year	1971; 71
     * M	年中的月份	Month	July; Jul; 07
     * w	年中的周数	Number	13
     * W	月份中的周数	Number	3
     * D	年中的天数	Number	232
     * d	月份中的天数	Number	10
     * F	月份中的星期	Number	2
     * E	星期中的天数	Text	Tuesday; Tue
     * a	Am/pm 标记	Text	PM
     * H	一天中的小时数（0-23）	Number	12
     * h	am/pm 中的小时数（1-12）	Number	12
     * k	一天中的小时数（1-24）	Number	24
     * K	am/pm 中的小时数（0-11）	Number	0
     * m	小时中的分钟数	Number	30
     * s	分钟中的秒数	Number	55
     * S	毫秒数	Number	978
     * Z	时区	RFC 822 time zone	-0800
     * z	时区	General time zone	Pacific Standard Time; PST; GMT-08:00
     * <p>
     * HH:mm    15:44
     * h:mm a    3:44 下午
     * HH:mm z    15:44 CST
     * HH:mm Z    15:44 +0800
     * HH:mm zzzz    15:44 中国标准时间
     * HH:mm:ss    15:44:40
     * yyyy-MM-dd    2016-08-12
     * yyyy-MM-dd HH:mm    2016-08-12 15:44
     * yyyy-MM-dd HH:mm:ss    2016-08-12 15:44:40
     * yyyy-MM-dd HH:mm:ss zzzz    2016-08-12 15:44:40 中国标准时间
     * EEEE yyyy-MM-dd HH:mm:ss zzzz    星期五 2016-08-12 15:44:40 中国标准时间
     * yyyy-MM-dd HH:mm:ss.SSSZ    2016-08-12 15:44:40.461+0800
     * yyyy-MM-dd'T'HH:mm:ss.SSSZ    2016-08-12T15:44:40.461+0800
     * yyyy.MM.dd G 'at' HH:mm:ss z    2016.08.12 公元 at 15:44:40 CST
     * K:mm a    3:44 下午
     * EEE, MMM d, ''yy    星期五, 八月 12, '16
     * hh 'o''clock' a, zzzz    03 o'clock 下午, 中国标准时间
     * yyyyy.MMMMM.dd GGG hh:mm aaa    02016.八月.12 公元 03:44 下午
     * EEE, d MMM yyyy HH:mm:ss Z    星期五, 12 八月 2016 15:44:40 +0800
     * yyMMddHHmmssZ    160812154440+0800
     * yyyy-MM-dd'T'HH:mm:ss.SSSZ    2016-08-12T15:44:40.461+0800
     * EEEE 'DATE('yyyy-MM-dd')' 'TIME('HH:mm:ss')' zzzz    星期五 DATE(2016-08-12) TIME(15:44:40) 中国标准时间
     * 注意SimpleDateFormat不是线程安全的
     */

    private static final String DEF_FORMAT = "yyyy-MM-dd HH:mm:ss";

    private static final String[] WEEK = new String[]{"星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六"};

    /**
     * 获取当前时间
     *
     * @param format
     * @return
     */
    public static String getCurrentTime(String format) {
        return new SimpleDateFormat(format).format(new Date());
    }

    /**
     * 当前时间戳
     *
     * @return
     */
    public static long getCurrentTimeMillis() {
        return Calendar.getInstance().getTimeInMillis();
    }

    /**
     * 将时间戳转为时间字符串
     *
     * @param milliseconds 毫秒时间戳
     * @param format       时间格式
     * @return 时间字符串
     */
    public static String timestamp2TimeString(long milliseconds, String format) {
        return new SimpleDateFormat(format).format(new Date(milliseconds));
    }

    /**
     * 将时间字符串转为时间戳
     *
     * @param time   时间字符串
     * @param format 时间格式
     * @return 毫秒时间戳
     */
    public static long timeString2Timestamp(String time, String format) {
        try {
            return new SimpleDateFormat(format).parse(time).getTime();
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return -1;
    }

    /**
     * 将Date类型转为时间戳
     *
     * @param time Date类型时间
     * @return 毫秒时间戳
     */
    public static long date2Timestamp(Date time) {
        return time.getTime();
    }

    /**
     * 获取时间戳
     *
     * @param year        实际年份
     * @param month       0--11
     * @param day         1--31
     * @param hour        0--23
     * @param minute      0--59
     * @param second      0--59
     * @param millisecond 3位小数
     * @return
     */
    public static long getTimestamp(int year, int month, int day, int hour, int minute, int second, int millisecond) {
        Calendar c = Calendar.getInstance();
        c.set(Calendar.YEAR, year);
        c.set(Calendar.MONTH, month);
        c.set(Calendar.DAY_OF_MONTH, day);
        c.set(Calendar.HOUR_OF_DAY, hour);
        c.set(Calendar.MINUTE, minute);
        c.set(Calendar.SECOND, second);
        c.set(Calendar.MILLISECOND, millisecond);
        return c.getTimeInMillis();
    }

    /**
     * 时间戳转成时间
     *
     * @param timestamp
     * @return [年，月，日，时，分，秒，毫秒，星期]
     */
    public static List<Integer> getTimeFromTimeStamp(long timestamp) {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(timestamp);
        List<Integer> timeList = new ArrayList<>();
        timeList.add(cal.get(Calendar.YEAR));
        timeList.add(cal.get(Calendar.MONTH));
        timeList.add(cal.get(Calendar.DAY_OF_MONTH));
        timeList.add(cal.get(Calendar.HOUR_OF_DAY));
        timeList.add(cal.get(Calendar.MINUTE));
        timeList.add(cal.get(Calendar.SECOND));
        timeList.add(cal.get(Calendar.MILLISECOND));
        timeList.add(cal.get(Calendar.DAY_OF_WEEK));
        return timeList;
    }

    public static boolean isAm(final Date date) {
        return getValueByCalendarField(date, Calendar.AM_PM) == 0;
    }

    /**
     * 获取星期几
     *
     * @param timestamp
     * @return
     */
    public static String getWeekName(long timestamp) {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(timestamp);
        int weak = cal.get(Calendar.DAY_OF_WEEK);
        if (weak > 0 && weak <= WEEK.length) {
            return WEEK[weak - 1];
        }
        return "";
    }

    /**
     * 转换成时分秒
     * @param distanceMillis
     * @return
     */
    public static String getDisplayHMS(long distanceMillis) {
        long second = distanceMillis / 1000;
        return String.format("%02d:%02d:%02d", (second / 3600), (second / 60 % 60), (second % 60));
    }

    /**
     * 设置系统时间（需要root权限）
     *
     * @param timestamp
     */
    public static void setSystemDate(long timestamp) {
        Log.i(TAG, "setSystemDate: ");
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            String dateTime = timestamp2TimeString(timestamp, "MMddHHmmyyyy.ss");
            ShellUtils.execCommand(new String[]{"setprop persist.sys.timezone Asia/Shanghai",
                    "su 0 toybox date " + dateTime
            }, true, false);
        }
        else {
            String dateTime = timestamp2TimeString(timestamp, "yyyyMMdd.HHmmss");
            ShellUtils.execCommand(new String[]{"setprop persist.sys.timezone Asia/Shanghai",
                    "su 0 toolbox date -s " + dateTime
            }, true, false);
        }
    }

    public static int getValueByCalendarField(final Date date, final int field) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.get(field);
    }
}
