package com.hx.lib_common.utils;

import android.text.TextUtils;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/4/2 13:51
 * Describe:
 */
public class FileUtils {

    /**
     * 同步文件到磁盘
     *
     * @param stream
     * @return
     */
    public static boolean sync(FileOutputStream stream) {
        try {
            if (stream != null) {
                stream.getFD().sync();
            }
            return true;
        } catch (IOException e) {

        }
        return false;
    }

    /**
     * 是否是文件夹
     *
     * @param dirPath
     * @return
     */
    public static boolean isDir(final String dirPath) {
        return isDir(new File(dirPath));
    }

    /**
     * 否是文件夹
     *
     * @param file
     * @return
     */
    public static boolean isDir(final File file) {
        return file != null && file.exists() && file.isDirectory();
    }

    /**
     * 是否是文件
     *
     * @param filePath
     * @return
     */
    public static boolean isFile(final String filePath) {
        return isFile(new File(filePath));
    }

    /**
     * 是否是文件
     *
     * @param file
     * @return
     */
    public static boolean isFile(final File file) {
        return file != null && file.exists() && file.isFile();
    }

    /**
     * 文件（夹）是否存在
     *
     * @param fileName
     * @return
     */
    public static boolean isFileExist(String fileName) {
        return isFileExist(new File(fileName));
    }

    /**
     * 文件（夹）是否存在
     *
     * @param file
     * @return
     */
    public static boolean isFileExist(File file) {
        return file != null && file.exists();
    }

    /**
     * 文件夹不存在就创建（包括父目录）
     *
     * @param dirPath
     * @return
     */
    public static boolean createOrExistsDir(final String dirPath) {
        return createOrExistsDir(new File(dirPath));
    }

    /**
     * 文件夹不存在就创建（包括父目录）
     *
     * @param file
     * @return
     */
    public static boolean createOrExistsDir(final File file) {
        return file != null && (file.exists() ? file.isDirectory() : file.mkdirs());
    }

    /**
     * 如果文件不存在则创建新文件，否则不做其它事
     *
     * @param filePath
     * @return
     */
    public static boolean createOrExistsFile(final String filePath) {
        return createOrExistsFile(new File(filePath));
    }

    /**
     * 如果文件不存在则创建新文件，否则不做其它事
     *
     * @param file
     * @return
     */
    public static boolean createOrExistsFile(final File file) {
        if (file == null) return false;
        if (file.exists()) return file.isFile();
        if (!createOrExistsDir(file.getParentFile())) return false;
        try {
            return file.createNewFile();
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        }
    }

    /**
     * 创建一个新的文件（如果文件存在会删除）
     *
     * @param filePath
     * @return
     */
    public static boolean createFullNewFile(final String filePath) {
        return createFullNewFile(new File(filePath));
    }

    /**
     * 创建一个新的文件（如果文件存在会删除）
     *
     * @param file
     * @return
     */
    public static boolean createFullNewFile(final File file) {
        if (file == null) return false;
        if (file.exists() && !file.delete()) return false;
        if (!createOrExistsDir(file.getParentFile())) return false;
        try {
            return file.createNewFile();
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        }
    }

    /**
     * 删除文件（文件夹目录下全部文件）
     *
     * @param filePath
     * @return
     */
    public static boolean delete(final String filePath) {
        return delete(new File(filePath));
    }

    /**
     * 删除文件（文件夹目录下全部文件）
     *
     * @param file
     * @return
     */
    public static boolean delete(final File file) {
        if (file == null) return false;
        if (file.isDirectory()) {
            return deleteDir(file);
        }
        return deleteFile(file);
    }

    /**
     * 删除文件夹
     *
     * @param dir
     * @return
     */
    private static boolean deleteDir(final File dir) {
        if (dir == null) return false;
        // dir doesn't exist then return true
        if (!dir.exists()) return true;
        // dir isn't a directory then return false
        if (!dir.isDirectory()) return false;
        File[] files = dir.listFiles();
        if (files != null && files.length != 0) {
            for (File file : files) {
                if (file.isFile()) {
                    if (!file.delete()) return false;
                } else if (file.isDirectory()) {
                    if (!deleteDir(file)) return false;
                }
            }
        }
        return dir.delete();
    }

    /**
     * 删除文件
     *
     * @param file
     * @return
     */
    private static boolean deleteFile(final File file) {
        return file != null && (!file.exists() || file.isFile() && file.delete());
    }

    /**
     * 删除目录下所有文件
     * @param dirPath
     * @return
     */
    public static boolean deleteFilesInDir(final String dirPath) {
        return deleteFilesInDir(new File(dirPath));
    }

    /**
     * 删除目录下所有文件
     * @param dir
     * @return
     */
    public static boolean deleteFilesInDir(final File dir) {
        return deleteFilesInDirWithFilter(dir, new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                return pathname.isFile();
            }
        });
    }


    /**
     * 删除指定后缀的文件名(需要包括“.”，如：.txt)
     * @param dirPath
     * @param extension
     * @return
     */
    public static boolean deleteFilesInDirWithExtension(final String dirPath, final String extension) {
        return deleteFilesInDirWithExtension(new File(dirPath), extension);
    }


    /**
     * 删除指定后缀的文件名(需要包括“.”，如：.txt)
     * @param dir
     * @param extension
     * @return
     */
    public static boolean deleteFilesInDirWithExtension(final File dir, final String extension){
        return deleteFilesInDirWithFilter(dir, new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                return pathname.isFile() && pathname.getName().endsWith(extension);
            }
        });
    }

    /**
     * 通过过滤器删除目录下的文件
     * @param dirPath
     * @param filter
     * @return
     */
    public static boolean deleteFilesInDirWithFilter(final String dirPath,
                                                     final FileFilter filter) {
        return deleteFilesInDirWithFilter(new File(dirPath), filter);
    }

    /**
     * 通过过滤器删除目录下的文件
     * @param dir
     * @param filter
     * @return
     */
    public static boolean deleteFilesInDirWithFilter(final File dir, final FileFilter filter) {
        if (dir == null || filter == null) return false;
        // dir doesn't exist then return true
        if (!dir.exists()) return true;
        // dir isn't a directory then return false
        if (!dir.isDirectory()) return false;
        File[] files = dir.listFiles();
        if (files != null && files.length != 0) {
            for (File file : files) {
                if (filter.accept(file)) {
                    if (file.isFile()) {
                        if (!file.delete()) return false;
                    } else if (file.isDirectory()) {
                        if (!deleteDir(file)) return false;
                    }
                }
            }
        }
        return true;
    }

    /**
     * 删除目录下的所有文件（不包括最顶层目录）
     *
     * @param dirPath
     * @return
     */
    public static boolean deleteFilesUnderDir(final String dirPath) {
        return deleteFilesUnderDir(new File(dirPath));
    }

    /**
     * 删除目录下的所有文件（不包括最顶层目录）
     *
     * @param dir
     * @return
     */
    public static boolean deleteFilesUnderDir(final File dir) {
        if (dir == null) return false;
        if (!dir.exists()) return true;
        if (!dir.isDirectory()) return false;
        File[] files = dir.listFiles();
        if (files != null && files.length != 0) {
            for (File file : files) {
                if (file.isFile()) {
                    if (!file.delete()) return false;
                } else if (file.isDirectory()) {
                    if (!deleteDir(file)) return false;
                }
            }
        }
        return true;
    }

    /**
     * 复制文件、文件夹
     *
     * @param srcPath
     * @param destPath
     * @return
     */
    public static boolean copy(final String srcPath,
                               final String destPath) {
        return copy(new File(srcPath), new File(destPath), null);
    }

    /**
     * 复制文件、文件夹
     *
     * @param srcPath
     * @param destPath
     * @param listener
     * @return
     */
    public static boolean copy(final String srcPath,
                               final String destPath,
                               final OnReplaceListener listener) {
        return copy(new File(srcPath), new File(destPath), listener);
    }

    /**
     * 复制文件、文件夹
     *
     * @param src
     * @param dest
     * @return
     */
    public static boolean copy(final File src,
                               final File dest) {
        return copy(src, dest, null);
    }

    /**
     * 复制文件、文件夹
     *
     * @param src
     * @param dest
     * @param listener
     * @return
     */
    public static boolean copy(final File src,
                               final File dest,
                               final OnReplaceListener listener) {
        if (src == null) return false;
        if (src.isDirectory()) {
            return copyDir(src, dest, listener);
        }
        return copyFile(src, dest, listener);
    }


    /**
     * 复制文件夹
     *
     * @param srcDir
     * @param destDir
     * @param listener
     * @return
     */
    public static boolean copyDir(final File srcDir,
                                  final File destDir,
                                  final OnReplaceListener listener) {
        return copyOrMoveDir(srcDir, destDir, listener, false);
    }

    /**
     * 复制文件
     *
     * @param srcFile
     * @param destFile
     * @param listener
     * @return
     */
    public static boolean copyFile(final File srcFile,
                                   final File destFile,
                                   final OnReplaceListener listener) {
        return copyOrMoveFile(srcFile, destFile, listener, false);
    }

    /**
     * 移动文件（文件夹）
     * @param srcPath
     * @param destPath
     * @return
     */
    public static boolean move(final String srcPath,
                               final String destPath) {
        return move(new File(srcPath), new File(destPath), null);
    }

    /**
     * 移动文件（文件夹）
     *
     * @param srcPath
     * @param destPath
     * @param listener
     * @return
     */
    public static boolean move(final String srcPath,
                               final String destPath,
                               final OnReplaceListener listener) {
        return move(new File(srcPath), new File(destPath), listener);
    }

    /**
     * 移动文件（文件夹）
     *
     * @param src
     * @param dest
     * @return
     */
    public static boolean move(final File src,
                               final File dest) {
        return move(src, dest, null);
    }

    /**
     * 移动文件（文件夹）
     *
     * @param src
     * @param dest
     * @param listener
     * @return
     */
    public static boolean move(final File src,
                               final File dest,
                               final OnReplaceListener listener) {
        if (src == null) return false;
        if (src.isDirectory()) {
            return moveDir(src, dest, listener);
        }
        return moveFile(src, dest, listener);
    }

    /**
     * 移动文件夹
     *
     * @param srcDir
     * @param destDir
     * @param listener
     * @return
     */
    public static boolean moveDir(final File srcDir,
                                  final File destDir,
                                  final OnReplaceListener listener) {
        return copyOrMoveDir(srcDir, destDir, listener, true);
    }

    /**
     * 移动文件
     *
     * @param srcFile
     * @param destFile
     * @param listener
     * @return
     */
    public static boolean moveFile(final File srcFile,
                                   final File destFile,
                                   final OnReplaceListener listener) {
        return copyOrMoveFile(srcFile, destFile, listener, true);
    }

    /**
     * 复制、移动文件夹
     *
     * @param srcDir
     * @param destDir
     * @param listener
     * @param isMove
     * @return
     */
    private static boolean copyOrMoveDir(final File srcDir,
                                         final File destDir,
                                         final OnReplaceListener listener,
                                         final boolean isMove) {
        if (srcDir == null || destDir == null) return false;
        if (!srcDir.exists() || !srcDir.isDirectory()) return false;
        // destDir's path locate in srcDir's path then return false
        String srcPath = srcDir.getPath() + File.separator;
        String destPath = destDir.getPath() + File.separator;
        if (destPath.contains(srcPath)) return false;
        if (!createOrExistsDir(destDir)) return false;
        File[] files = srcDir.listFiles();
        for (File file : files) {
            File oneDestFile = new File(destPath + file.getName());
            if (file.isFile()) {
                if (!copyOrMoveFile(file, oneDestFile, listener, isMove)) return false;
            } else if (file.isDirectory()) {
                if (!copyOrMoveDir(file, oneDestFile, listener, isMove)) return false;
            }
        }
        return !isMove || deleteDir(srcDir);
    }

    /**
     * 复制或者移动文件
     *
     * @param srcFile
     * @param destFile
     * @param listener
     * @param isMove
     * @return
     */
    private static boolean copyOrMoveFile(final File srcFile,
                                          final File destFile,
                                          final OnReplaceListener listener,
                                          final boolean isMove) {
        if (srcFile == null || destFile == null) return false;
        // srcFile equals destFile then return false
        if (srcFile.equals(destFile)) return false;
        // srcFile doesn't exist or isn't a file then return false
        if (!srcFile.exists() || !srcFile.isFile()) return false;
        if (destFile.exists()) {
            if (listener == null || listener.onReplace(srcFile, destFile)) {// require delete the old file
                if (!destFile.delete()) {// unsuccessfully delete then return false
                    return false;
                }
            } else {
                return true;
            }
        }
        if (!createOrExistsDir(destFile.getParentFile())) return false;
        try {
            return FileIoUtils.writeFileFromIS(destFile, new FileInputStream(srcFile))
                    && !(isMove && !deleteFile(srcFile));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return false;
        }
    }

    /**
     * 获取文件名（没后缀）
     * @param file
     * @return
     */
    public static String getFileNameNoExtension(final File file) {
        if (file == null) return "";
        return getFileNameNoExtension(file.getPath());
    }

    /**
     * 获取文件名（没后缀）
     * @param filePath
     * @return
     */
    public static String getFileNameNoExtension(final String filePath) {
        if (TextUtils.isEmpty(filePath)) return "";
        int lastPoi = filePath.lastIndexOf('.');
        int lastSep = filePath.lastIndexOf(File.separator);
        if (lastSep == -1) {
            return (lastPoi == -1 ? filePath : filePath.substring(0, lastPoi));
        }
        if (lastPoi == -1 || lastSep > lastPoi) {
            return filePath.substring(lastSep + 1);
        }
        return filePath.substring(lastSep + 1, lastPoi);
    }


    public interface OnReplaceListener {
        boolean onReplace(File srcFile, File destFile);
    }
}
