package com.hx.lib_common.utils;

import android.util.Log;

import com.hx.lib_common.constant.MemoryConst;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.List;

/**
 * Copyright © 1997 - 2020 Gosuncn. All Rights Reserved
 * Created by huangxi on 2020/4/8 10:49
 * Describe:
 * 文件流工具类
 */
public class FileIoUtils {
    private static final String TAG = "FileIoUtils";
    private static int sBufferSize = 512 * MemoryConst.KB;

    /**
     * 从输入流写入到文件
     * @param filePath
     * @param is
     * @return
     */
    public static boolean writeFileFromIS(final String filePath,
                                          final InputStream is){
        return writeFileFromIS(new File(filePath), is, false, null);
    }

    /**
     * 从输入流写入到文件
     * @param file
     * @param is
     * @return
     */
    public static boolean writeFileFromIS(final File file,
                                          final InputStream is){
        return writeFileFromIS(file, is, false, null);
    }

    /**
     * 从输入流写入到文件
     *
     * @param filePath
     * @param is
     * @param append
     * @param listener
     * @return
     */
    public static boolean writeFileFromIS(final String filePath,
                                          final InputStream is,
                                          final boolean append,
                                          final OnProgressUpdateListener listener) {
        return writeFileFromIS(new File(filePath), is, append, listener);
    }

    /**
     * 从输入流写入到文件
     *
     * @param file
     * @param is
     * @param append
     * @param listener
     * @return
     */
    public static boolean writeFileFromIS(final File file,
                                          final InputStream is,
                                          final boolean append,
                                          final OnProgressUpdateListener listener) {
        if (is == null || !FileUtils.createOrExistsFile(file)) {
            Log.e(TAG, "create file <" + file + "> failed.");
            return false;
        }
        OutputStream os = null;
        try {
            os = new BufferedOutputStream(new FileOutputStream(file, append), sBufferSize);
            if (listener == null) {
                byte[] data = new byte[sBufferSize];
                for (int len; (len = is.read(data)) != -1; ) {
                    os.write(data, 0, len);
                }
            } else {
                double totalSize = is.available();
                int curSize = 0;
                listener.onProgressUpdate(0);
                byte[] data = new byte[sBufferSize];
                for (int len; (len = is.read(data)) != -1; ) {
                    os.write(data, 0, len);
                    curSize += len;
                    listener.onProgressUpdate(curSize / totalSize);
                }
            }
            return true;
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        } finally {
            CloseUtils.closeIO(is, os);
        }
    }

    /**
     * 将数组写入文件
     *
     * @param filePath
     * @param bytes
     * @param append
     * @param listener
     * @return
     */
    public static boolean writeFileFromBytes(final String filePath,
                                             final byte[] bytes,
                                             final boolean append,
                                             final OnProgressUpdateListener listener) {
        return writeFileFromBytes(new File(filePath), bytes, append, listener);
    }

    /**
     * 将数组写入文件
     *
     * @param file
     * @param bytes
     * @param append
     * @param listener
     * @return
     */
    public static boolean writeFileFromBytes(final File file,
                                             final byte[] bytes,
                                             final boolean append,
                                             final OnProgressUpdateListener listener) {
        if (bytes == null) return false;
        return writeFileFromIS(file, new ByteArrayInputStream(bytes), append, listener);
    }

    /**
     * 通过FileChannel将数组写入文件
     *
     * @param filePath
     * @param bytes
     * @param append
     * @param isForce
     * @return
     */
    public static boolean writeFileFromBytesByChannel(final String filePath,
                                                      final byte[] bytes,
                                                      final boolean append,
                                                      final boolean isForce) {
        return writeFileFromBytesByChannel(new File(filePath), bytes, append, isForce);
    }

    /**
     * 通过FileChannel将数组写入文件
     *
     * @param file
     * @param bytes
     * @param append
     * @param isForce 强制写入磁盘
     * @return
     */
    public static boolean writeFileFromBytesByChannel(final File file,
                                                      final byte[] bytes,
                                                      final boolean append,
                                                      final boolean isForce) {
        if (bytes == null || !FileUtils.createOrExistsFile(file)) {
            Log.e("FileIOUtils", "create file <" + file + "> failed.");
            return false;
        }
        FileChannel fc = null;
        FileOutputStream outputStream = null;
        try {
            outputStream = new FileOutputStream(file, append);
            fc = outputStream.getChannel();
            if (fc == null) {
                Log.e(TAG, "fc is null.");
                return false;
            }
            fc.position(fc.size());
            fc.write(ByteBuffer.wrap(bytes));
            if (isForce) fc.force(true);
            return true;
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        } finally {
            CloseUtils.closeIO(fc, outputStream);
        }
    }

    /**
     * 通过Channel的Map将数组写入文件
     *
     * @param file
     * @param bytes
     * @param append
     * @param isForce
     * @return
     */
    public static boolean writeFileFromBytesByMap(final File file,
                                                  final byte[] bytes,
                                                  final boolean append,
                                                  final boolean isForce) {
        if (bytes == null || !FileUtils.createOrExistsFile(file)) {
            Log.e("FileIOUtils", "create file <" + file + "> failed.");
            return false;
        }
        try {
            FileChannel fc = null;
            FileOutputStream fos = new FileOutputStream(file, append);
            try {
                fc = fos.getChannel();
                if (fc == null) {
                    Log.e("FileIOUtils", "fc is null.");
                    return false;
                }
                MappedByteBuffer mbb = fc.map(FileChannel.MapMode.READ_WRITE, fc.size(), bytes.length);
                mbb.put(bytes);
                if (isForce) mbb.force();
                return true;
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            } finally {
                CloseUtils.closeIO(fos, fc);
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 将字符串写入文件
     *
     * @param filePath
     * @param content
     * @param append
     * @return
     */
    public static boolean writeFileFromString(final String filePath,
                                              final String content,
                                              final boolean append) {
        return writeFileFromString(new File(filePath), content, append);
    }

    /**
     * 将字符串写入文件
     *
     * @param file
     * @param content
     * @param append
     * @return
     */
    public static boolean writeFileFromString(final File file,
                                              final String content,
                                              final boolean append) {
        if (file == null || content == null) return false;
        if (!FileUtils.createOrExistsFile(file)) {
            Log.e(TAG, "create file <" + file + "> failed.");
            return false;
        }
        BufferedWriter bw = null;
        FileWriter writer = null;
        try {
            writer = new FileWriter(file, append);
            bw = new BufferedWriter(writer);
            bw.write(content);
            return true;
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        } finally {
            CloseUtils.closeIO(bw, writer);
        }
    }

    /**
     * 按行读文件
     *
     * @param filePath
     * @return
     */
    public static List<String> readFile2List(final String filePath) {
        return readFile2List(new File(filePath), 0, Integer.MAX_VALUE, null);
    }

    /**
     * 按行读文件
     *
     * @param filePath
     * @param charsetName
     * @return
     */
    public static List<String> readFile2List(final String filePath, final String charsetName) {
        return readFile2List(new File(filePath), 0, Integer.MAX_VALUE, charsetName);
    }

    /**
     * 按行读文件
     *
     * @param file
     * @return
     */
    public static List<String> readFile2List(final File file) {
        return readFile2List(file, 0, Integer.MAX_VALUE, null);
    }

    /**
     * 按行读文件
     *
     * @param file
     * @param charsetName
     * @return
     */
    public static List<String> readFile2List(final File file, final String charsetName) {
        return readFile2List(file, 0, Integer.MAX_VALUE, charsetName);
    }

    /**
     * 按行读文件
     *
     * @param file
     * @param st          开始行
     * @param end         结束行
     * @param charsetName
     * @return
     */
    public static List<String> readFile2List(final File file,
                                             final int st,
                                             final int end,
                                             final String charsetName) {
        if (!FileUtils.isFileExist(file)) return null;
        if (st > end) return null;
        BufferedReader reader = null;
        FileInputStream inputStream = null;
        InputStreamReader inputReader = null;
        try {
            String line;
            int curLine = 1;
            List<String> list = new ArrayList<>();
            inputStream = new FileInputStream(file);
            if (StringUtils.isEmpty(charsetName)) {
                inputReader = new InputStreamReader(inputStream);
            } else {
                inputReader = new InputStreamReader(inputStream, charsetName);
            }
            reader = new BufferedReader(inputReader);
            while ((line = reader.readLine()) != null) {
                if (curLine > end) break;
                if (st <= curLine && curLine <= end) list.add(line);
                ++curLine;
            }
            return list;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } finally {
            CloseUtils.closeIO(reader, inputReader, inputStream);
        }
    }

    /**
     * 读文件到String
     *
     * @param filePath
     * @return
     */
    public static String readFile2String(final String filePath) {
        return readFile2String(new File(filePath), null);
    }

    /**
     * 读文件到String
     *
     * @param filePath
     * @param charsetName
     * @return
     */
    public static String readFile2String(final String filePath, final String charsetName) {
        return readFile2String(new File(filePath), charsetName);
    }

    /**
     * 读文件到String
     *
     * @param file
     * @return
     */
    public static String readFile2String(final File file) {
        return readFile2String(file, null);
    }

    /**
     * 读文件到String
     *
     * @param file
     * @param charsetName
     * @return
     */
    public static String readFile2String(final File file, final String charsetName) {
        byte[] contents = readFile2Bytes(file);
        if (contents == null) return null;
        if (StringUtils.isEmpty(charsetName)) {
            return new String(contents);
        } else {
            try {
                return new String(contents, charsetName);
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
                return "";
            }
        }
    }

    /**
     * 读取文件到数组
     *
     * @param filePath
     * @return
     */
    public static byte[] readFile2Bytes(final String filePath) {
        return readFile2Bytes(new File(filePath), null);
    }

    /**
     * 读取文件到数组
     *
     * @param file
     * @return
     */
    public static byte[] readFile2Bytes(final File file) {
        return readFile2Bytes(file, null);
    }

    /**
     * 读取文件到数组
     *
     * @param filePath
     * @param listener
     * @return
     */
    public static byte[] readFile2Bytes(final String filePath,
                                        final OnProgressUpdateListener listener) {
        return readFile2Bytes(new File(filePath), listener);
    }

    /**
     * 读取文件到数组
     *
     * @param file
     * @param listener
     * @return
     */
    public static byte[] readFile2Bytes(final File file,
                                        final OnProgressUpdateListener listener) {
        if (!FileUtils.isFileExist(file)) return null;
        try {
            ByteArrayOutputStream os = null;
            FileInputStream fis = new FileInputStream(file);
            InputStream is = new BufferedInputStream(fis, sBufferSize);
            try {
                os = new ByteArrayOutputStream();
                byte[] b = new byte[sBufferSize];
                int len;
                if (listener == null) {
                    while ((len = is.read(b, 0, sBufferSize)) != -1) {
                        os.write(b, 0, len);
                    }
                } else {
                    double totalSize = is.available();
                    int curSize = 0;
                    listener.onProgressUpdate(0);
                    while ((len = is.read(b, 0, sBufferSize)) != -1) {
                        os.write(b, 0, len);
                        curSize += len;
                        listener.onProgressUpdate(curSize / totalSize);
                    }
                }
                return os.toByteArray();
            } catch (IOException e) {
                e.printStackTrace();
                return null;
            } finally {
                CloseUtils.closeIO(fis, is, os);
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 通过Channel读文件
     *
     * @param filePath
     * @return
     */
    public static byte[] readFile2BytesByChannel(final String filePath) {
        return readFile2BytesByChannel(new File(filePath));
    }

    /**
     * 通过Channel读文件
     *
     * @param file
     * @return
     */
    public static byte[] readFile2BytesByChannel(final File file) {
        if (!FileUtils.isFileExist(file)) return null;
        try {
            FileChannel fc = null;
            RandomAccessFile accessFile = new RandomAccessFile(file, "r");
            try {
                fc = accessFile.getChannel();
                if (fc == null) {
                    Log.e(TAG, "fc is null.");
                    return new byte[0];
                }
                ByteBuffer byteBuffer = ByteBuffer.allocate((int) fc.size());
                while (true) {
                    if (!((fc.read(byteBuffer)) > 0)) break;
                }
                return byteBuffer.array();
            } catch (IOException e) {
                e.printStackTrace();
                return null;
            } finally {
                CloseUtils.closeIO(fc, accessFile);
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 通过Channel的Map读入文件
     *
     * @param filePath
     * @return
     */
    public static byte[] readFile2BytesByMap(final String filePath) {
        return readFile2BytesByMap(new File(filePath));
    }

    /**
     * 通过Channel的Map读入文件
     *
     * @param file
     * @return
     */
    public static byte[] readFile2BytesByMap(final File file) {
        if (!FileUtils.isFileExist(file)) return null;
        try {
            FileChannel fc = null;
            RandomAccessFile accessFile = new RandomAccessFile(file, "r");
            try {
                fc = new RandomAccessFile(file, "r").getChannel();
                if (fc == null) {
                    Log.e("FileIOUtils", "fc is null.");
                    return new byte[0];
                }
                int size = (int) fc.size();
                MappedByteBuffer mbb = fc.map(FileChannel.MapMode.READ_ONLY, 0, size).load();
                byte[] result = new byte[size];
                mbb.get(result, 0, size);
                return result;
            } catch (IOException e) {
                e.printStackTrace();
                return null;
            } finally {
                CloseUtils.closeIO(accessFile, fc);
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

    public interface OnProgressUpdateListener {
        void onProgressUpdate(double progress);
    }
}
