package io.gitee.mrxangel.producer;

import io.gitee.mrxangel.config.PulsarProperties;
import io.gitee.mrxangel.enums.BatcherBuilderEnum;
import io.gitee.mrxangel.enums.CompressionTypeEnum;
import org.apache.pulsar.client.api.Producer;
import org.apache.pulsar.client.api.PulsarClient;
import org.apache.pulsar.client.api.PulsarClientException;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.annotation.Configuration;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;

@Component
@Configuration
public class PulsarProducerHandler implements BeanPostProcessor {
    @Resource
    private PulsarProperties pulsarProperties;
   /* @Resource*/
    private  PulsarClient pulsarClient;
    public PulsarProducerHandler(PulsarClient pulsarClient) {
        this.pulsarClient = pulsarClient;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        Arrays.stream(bean.getClass().getFields()).filter(v -> v.isAnnotationPresent(PulsarProducer.class)).forEach(method->{
            PulsarProducer annotation = method.getAnnotation(PulsarProducer.class);
            initProducer( annotation.topic());
        });

        return bean;
    }

    /**
     * init producer
     * @param topic 主题
     */
    public void initProducer(String topic){
        try {
            Producer<byte[]> producer = pulsarClient.newProducer()
                    .topic(topic)
                    .enableBatching(pulsarProperties.getEnableBatching())//是否开启批量处理消息，默认true,需要注意的是enableBatching只在异步发送sendAsync生效，同步发送send失效。因此建议生产环境若想使用批处理，则需使用异步发送，或者多线程同步发送
                    .compressionType(CompressionTypeEnum.of(pulsarProperties.getCompressionType()))//消息压缩（四种压缩方式：LZ4，ZLIB，ZSTD，SNAPPY），consumer端不用做改动就能消费，开启后大约可以降低3/4带宽消耗和存储（官方测试）
                    .batchingMaxPublishDelay(pulsarProperties.getBatchingMaxPublishDelay(), TimeUnit.MILLISECONDS) //设置将对发送的消息进行批处理的时间段,10ms；可以理解为若该时间段内批处理成功，则一个batch中的消息数量不会被该参数所影响。
                    .sendTimeout(pulsarProperties.getSendTimeout(), TimeUnit.SECONDS)//设置发送超时0s；如果在sendTimeout过期之前服务器没有确认消息，则会发生错误。默认30s，设置为0代表无限制，建议配置为0
                    .batchingMaxMessages(pulsarProperties.getBatchingMaxMessages())//批处理中允许的最大消息数。默认1000
                    .maxPendingMessages(pulsarProperties.getMaxPendingMessages())//设置等待接受来自broker确认消息的队列的最大大小，默认1000
                    .blockIfQueueFull(pulsarProperties.getBlockIfQueueFull())//设置当消息队列中等待的消息已满时，Producer.send 和 Producer.sendAsync 是否应该block阻塞。默认为false，达到maxPendingMessages后send操作会报错，设置为true后，send操作阻塞但是不报错。建议设置为true
                    .roundRobinRouterBatchingPartitionSwitchFrequency(pulsarProperties.getRoundRobinRouterBatchingPartitionSwitchFrequency())//向不同partition分发消息的切换频率，默认10ms，可根据batch情况灵活调整
                    .batcherBuilder(BatcherBuilderEnum.of(pulsarProperties.getBatcherBuilder()))//key_Shared模式要用KEY_BASED,才能保证同一个key的message在一个batch里
                    .create();
            PulsarProducerFactory.addPulsarProducerFactory(topic,producer);
        } catch (PulsarClientException e) {
            e.printStackTrace();
        }
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) {
        return bean;
    }
}
