package com.lk.http.argument;

import com.lk.annotation.RpcParamRequired;
import com.lk.http.method.MethodValueInfo;
import com.thoughtworks.paranamer.BytecodeReadingParanamer;
import com.thoughtworks.paranamer.Paranamer;
import org.apache.commons.collections.CollectionUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * @author lk
 */
public class ArgumentNameDiscovererUtils {

    static Paranamer paranamer = new BytecodeReadingParanamer();
    static Map<Method, List<MethodValueInfo>> argumentNamesResolverCacheMap = new ConcurrentHashMap<Method, List<MethodValueInfo>>();

    /**
     * 根据方法获取方法需要的参数列表
     *
     * @param method
     * @return
     */
    public static List<MethodValueInfo> getCachedArgumentNames(Method method) {
        // 先从缓存中获取
        List<MethodValueInfo> methodValueInfoList = argumentNamesResolverCacheMap.get(method);
        if (CollectionUtils.isEmpty(methodValueInfoList)) {
            String[] results = paranamer.lookupParameterNames(method);
            methodValueInfoList = new ArrayList<MethodValueInfo>();
            for (String result : results) {
                methodValueInfoList.add(new MethodValueInfo(result, false, null));
            }
            // 更新参数
            updateNameValues(method, methodValueInfoList);
            // 加入本地缓存
            argumentNamesResolverCacheMap.put(method, methodValueInfoList);
            return methodValueInfoList;
        }
        return methodValueInfoList;
    }

    /**
     * 通过注解更新参数相关信息
     *
     * @param method
     * @param methodValueInfoList
     */
    private static void updateNameValues(Method method, List<MethodValueInfo> methodValueInfoList) {
        Annotation[][] parameterAnnotations = method.getParameterAnnotations();
        Class<?>[] parameterTypes = method.getParameterTypes();
        Type[] genericParameterTypes = method.getGenericParameterTypes();
        if (methodValueInfoList.size() != parameterTypes.length) {
            throw new IllegalArgumentException("注解列表个数与方法数不匹配");
        }
        //设置clazz类型
        int idx = 0;
        for (Class<?> parameterType : parameterTypes) {
            methodValueInfoList.get(idx).setClazz(parameterType);
            methodValueInfoList.get(idx).setType(genericParameterTypes[idx]);
            idx++;
        }
        //通过注解更新参数名称
        int index = 0;
        for (Annotation[] parameterAnnotation : parameterAnnotations) {
            if (parameterAnnotation.length == 0) {
                index++;
                continue;
            }
            MethodValueInfo methodValueInfo = methodValueInfoList.get(index);
            for (Annotation annotation : parameterAnnotation) {
                if (annotation instanceof RpcParamRequired) {
                    RpcParamRequired rpcParamRequired = (RpcParamRequired) annotation;
                    methodValueInfo.setRequired(rpcParamRequired.required());
                    methodValueInfo.setDefaultValue(rpcParamRequired.defaultValue());
                }
            }
            index++;
        }
    }

}
