package com.lk.http;

import com.lk.annotation.RpcMethod;
import com.lk.exceptions.MethodNotFindException;
import com.lk.exceptions.MoreThanOneMethodException;
import com.lk.http.argument.ParseParametersUtils;
import com.lk.http.context.SpringContextUtil;
import com.lk.http.entity.ContentType;
import com.lk.response.JsonResponseCode;
import com.lk.response.ResponseUtils;
import org.apache.commons.lang.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;


/**
 * @author lk
 */
public class RPCEntranceDispatcher {

    /**
     * 请求处理
     *
     * @param response
     * @param request
     * @param beanName
     * @param methodName
     * @throws Throwable
     */
    public void call(HttpServletResponse response, HttpServletRequest request, String beanName, String methodName) throws Throwable {
        // 校验并获取bean实例
        Object bean;
        try {
            bean = SpringContextUtil.getBean(beanName);
        } catch (Throwable throwable) {
            throw new Throwable("未找到对应的bean");
        }
        if (StringUtils.isBlank(beanName) || StringUtils.isBlank(methodName)) {
            ResponseUtils.outputError(response, "请求参数缺失，类名或者方法名找不到");
            return;
        }
        // 设置返回参数类型
//        response.setContentType(ContentType.APPLICATION_JSON);
        // 校验并获取方法
        Method method = null;
        method = findMethodOnlyByName(bean, methodName);
        if (method == null) {
            ResponseUtils.outputError(response, "can't find method:" + methodName);
            return;
        }

        //所有异常检测在这里
//        if (!check(getIsCsrfSwitch(), csrfKey, request, response, beanName, methodName)) {
//            LOG.warn("/web/j/" + beanName + "." + methodName + ".rpc exception:" + "异步请求调用目标前没有通过检查");
//            return;
//        }

        // 处理参数并执行方法
        if (StringUtils.contains(request.getContentType(), ContentType.APPLICATION_FORM_URLENCODED)) {
            proceedByParamInput(request, response, bean, method);
        } else {
            ResponseUtils.outputError(response, "can't recognize contentType:" + request.getContentType() + ", only support:" + Collections.singletonList(ContentType.APPLICATION_FORM_URLENCODED));
        }
    }

    /**
     * 根据方法名寻找在指定类中寻找方法
     *
     * @param bean
     * @param methodName
     * @return
     */
    private Method findMethodOnlyByName(Object bean, String methodName) {
        Method[] methods = bean.getClass().getMethods();
        Method matchedMethod = null;
        int matched = 0;
        for (int i = 0; i < methods.length; i++) {
            // Check method name and access
            if (methods[i].getName().equals(methodName)) {
                if (methods[i].getAnnotation(RpcMethod.class) != null) {
                    matched++;
                    matchedMethod = methods[i];
                }
            }
        }
        if (matched > 1) {
            throw new MoreThanOneMethodException(JsonResponseCode.ERROR_METHOD_CANT_FIND, " find more than one!" + methodName);
        }
        if (matched < 1) {
            throw new MethodNotFindException(JsonResponseCode.ERROR_METHOD_CANT_FIND, " find no such method:" + methodName);
        }
        return matchedMethod;
    }

    /**
     * 处理传输的参数
     *
     * @param request
     * @param response
     * @param bean
     * @param method
     * @throws IOException
     */
    private void proceedByParamInput(HttpServletRequest request, HttpServletResponse response, Object bean, Method method) throws IOException {
        Object[] paramArray = null;
        try {
            // 解析参数列表
            List<Object> parametersForInput = ParseParametersUtils.parseParametersForURLEncode(method, request);
            paramArray = parametersForInput.toArray(new Object[parametersForInput.size()]);
            // 执行方法
            Object result = method.invoke(bean, paramArray);
            ResponseUtils.output(response, result);
        } catch (Exception e) {
            ResponseUtils.outputError(response, "请求失败，请稍后再试");
        }
    }

}
