/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.sqldata.mock.core;

import io.gitee.jaemon.sqldata.mock.MockException;
import io.gitee.jaemon.sqldata.mock.SqlConfig;
import io.gitee.jaemon.sqldata.mock.entity.Column;

import java.sql.*;
import java.util.ArrayList;
import java.util.List;

import static io.gitee.jaemon.sqldata.mock.core.SqlTemplate.*;

/**
 * 操作sql执行器
 *
 * @author Jaemon
 * @since 1.0
 */
public class SqlExecutor {
    protected static final SqlConfig sqlConfig = SqlConfig.sqlConfig();
    protected static Connection conn;
    static {
        try {
            Class.forName(sqlConfig.getDriver());
            conn = DriverManager.getConnection(sqlConfig.getUrl(), sqlConfig.getUname(), sqlConfig.getPassword());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static final String db = sqlConfig.getDb();

    /**
     * 获取db的所有表名
     *
     * @return
     *      表名集合
     * */
    public static List<String> tableNames() {
        List<String> tables = new ArrayList<>();
        try(
                Statement statement = conn.createStatement();
                ResultSet resultSet = statement.executeQuery(
                        String.format(DB_TABLES_SQL, db)
                )
        ) {
            while (resultSet.next()) {
                tables.add(resultSet.getString(1));
            }

        } catch (Exception ex) {
            throw new MockException("");
        }
        return tables;
    }


    /**
     * 获取db的所有字段信息
     *
     * @return
     *      字段信息集合
     * */
    public static List<Column> dbColumns() {
        List<Column> tables = new ArrayList<>();
        try(
                Statement statement = conn.createStatement();
                ResultSet resultSet = statement.executeQuery(
                        String.format(DB_COLUMNS_SQL, db)
                )
        ) {
            while (resultSet.next()) {
                tables.add(
                        new Column(
                                resultSet.getString(1),
                                resultSet.getString(2),
                                resultSet.getString(3)
                        )
                );
            }

        } catch (Exception ex) {
            throw new MockException("");
        } finally {
            try {
                conn.close();
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
        return tables;
    }

    /**
     * 获取表的所有字段
     *
     * @param tableName
     *          表名
     * @return
     *          表字段集合
     */
    public static List<String> tableColumns(String tableName) {
        String sql = String.format(TABLE_COLUMNS_SQL, db, tableName);
        List<String> columns = new ArrayList<>();

        try(
                Statement statement = conn.createStatement();
                ResultSet resultSet = statement.executeQuery(sql)
        ) {

            while (resultSet.next()) {
                columns.add(resultSet.getString(1));
            }

        } catch (Exception ex) {
            throw new MockException("");
        }
        return columns;
    }

    /**
     * 执行sql插入
     *
     * @param sql
     *      插入的sql语句
     * @return
     *      影响结果数
     */
    public static int insert(String sql) {
        try(
                Statement statement = conn.createStatement();
        ) {
            return statement.executeUpdate(sql);
        } catch (Exception e) {
            System.out.println(String.format("sql=%s, msg=%s.", sql, e.getMessage()));
            return 0;
        }
    }

    /**
     * 批量执行sql插入
     *
     * @param sqls
     *      批量sql语句集
     * @return
     *      影响结果数
     */
    public static int[] insertBatch(String... sqls) {
        try {
            conn.setAutoCommit(false);
        } catch (SQLException e) {
            throw new MockException("");
        }
        try(
                Statement statement = conn.createStatement();
        ) {
            for (String sql : sqls) {
                statement.addBatch(sql);
            }
            int[] count = statement.executeBatch();
            return count;
        } catch (Exception e) {
            System.out.println(String.format("msg=%s.", e.getMessage()));
            return new int[0];
        } finally {
            try {
                conn.setAutoCommit(true);
            } catch (SQLException e) {
                throw new MockException("");
            }
        }
    }

}