/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.sqldata.mock.core;

import io.gitee.jaemon.sqldata.mock.MockException;
import io.gitee.jaemon.sqldata.mock.TableColumnsHandler;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static io.gitee.jaemon.sqldata.mock.core.SqlExecutor.*;
import static io.gitee.jaemon.sqldata.mock.core.SqlExecutor.insertBatch;
import static io.gitee.jaemon.sqldata.mock.core.SqlTemplate.TRUNCATE_TABLE_SQL;

/**
 * 数据mock执行器
 *
 * @author Jaemon
 * @since 1.0
 */
public class SqlDataMockExecutor extends MockTemplate {
    public SqlDataMockExecutor(Class<? extends TableColumnsHandler> clazz) {
        super(clazz);
    }

    @Override
    List<String> tables() throws MockException {
        List<String> tables = new ArrayList<>();
        if (sqlConfig.getIncludeTables().isEmpty()) {
            tables.addAll(tableNames());
        } else {
            tables.addAll(sqlConfig.getIncludeTables());
        }

        List<String> excludeTables = sqlConfig.getExcludeTables();
        if (!excludeTables.isEmpty()) {
            return tables.stream().filter(e -> !excludeTables.contains(e)).collect(Collectors.toList());
        }

        return tables;
    }

    @Override
    void tableSql(String table) throws MockException {
        List<String> tableColumns = tableColumns(table);

        String columns = String.join(",", tableColumns);
        StringBuilder insertSql = new StringBuilder();
        insertSql.append(String.format(INSERT_PREFIX, table, columns));

        StringBuilder valueSql = new StringBuilder();
        for (int i = 0; i < sqlConfig.getRecordCount(); i++) {
            valueSql.append("(");
            for (String column : tableColumns) {
                TableColumnsHandler handler = ColumnHandlerFactory.get(column);
                Object value = null;
                if (handler != null) {
                    value = handler.generate(table);
                }
                if (String.class.isInstance(value)) {
                    valueSql.append("'" + value + "'").append(",");
                } else {
                    valueSql.append(value).append(",");
                }

            }
            valueSql.deleteCharAt(valueSql.length() - 1);
            valueSql.append("),");
        }
        valueSql.deleteCharAt(valueSql.length() - 1);

        insertSql.append(valueSql);
        if (sqlConfig.isTruncateTable()) {
            insertBatch(String.format(TRUNCATE_TABLE_SQL, table), insertSql.toString());
        } else {
            insertBatch(insertSql.toString());
        }
    }

    @Override
    public void close() throws Exception {
        conn.close();
    }
}