/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.sqldata.mock;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

/**
 * 数据库配置类
 *
 * @author Jaemon
 * @since 1.0
 */
public class SqlConfig {
    private static final Properties sqlProperties = new Properties();
    private static final String DEFAULT_SQL_CONFIG = "sqldatamock.properties";
    public static final int DEFAULT_RECORD_COUNT = 50;

    private static SqlConfig sqlConfig;
    private String driver = "com.mysql.jdbc.Driver";
    private String url = "jdbc:mysql://127.0.0.1:3306/mockdb?useUnicode=true&characterEncoding=utf-8&useSSL=false";
    private String uname = "root";
    private String password = "root";
    private String db;
    private boolean truncateTable = false;
    private int recordCount = 50;
    // 待处理的表集合, 如果不设置，默认库中所有表
    private List<String> includeTables = new ArrayList<>();
    // 排除表集合
    private List<String> excludeTables = new ArrayList<>();

    private SqlConfig() {
        try {
            sqlProperties.load(this.getClass().getClassLoader().getResourceAsStream(DEFAULT_SQL_CONFIG));

            driver = sqlProperties.getProperty("driver", driver);
            url = sqlProperties.getProperty("url", url);
            uname = sqlProperties.getProperty("username", uname);
            password = sqlProperties.getProperty("password", password);
            db = database(url);

            String truncatTableFlag = sqlProperties.getProperty("truncate-table", "false");
            if ("true".equals(truncatTableFlag)) {
                truncateTable = true;
            }

            String recordCountStr = sqlProperties.getProperty("record-count", "50");
            recordCount = Integer.parseInt(recordCountStr);

            recordCount = recordCount < 0 ? DEFAULT_RECORD_COUNT : recordCount > 1000 ? 1000 : recordCount;

            String includeTablesStr = sqlProperties.getProperty("include-tables");
            if (isNotEmpty(includeTablesStr)) {
                includeTables = Arrays.asList(includeTablesStr.split(","));
            }

            String excludeTablesStr = sqlProperties.getProperty("exclude-tables");
            if (isNotEmpty(excludeTablesStr)) {
                excludeTables = Arrays.asList(excludeTablesStr.split(","));
            }

            print("url=%s", url);
            print("username=%s", uname);
            print("password=%s", password);
            print("truncateTable=%s", truncateTable);
            print("recordCount=%s", recordCount);
        } catch (Exception ex) {
            print("配置文件异常");
        }
    }

    public static SqlConfig sqlConfig() {
        if (sqlConfig == null) {
            sqlConfig = new SqlConfig();
        }
        return sqlConfig;
    }

    private static String database(String url) {
        int start = url.lastIndexOf('/') + 1;
        int end = url.lastIndexOf('?');
        return url.substring(start, end);
    }

    private static boolean isEmpty(String str) {
        if (str == null || "".equals(str.trim())) {
            return true;
        }
        return false;
    }

    private static void print(String format, Object... args) {
        System.out.println(String.format(format, args));
    }

    private static boolean isNotEmpty(String str) {
        return !isEmpty(str);
    }

    public String getDriver() {
        return driver;
    }

    public String getUrl() {
        return url;
    }

    public String getUname() {
        return uname;
    }

    public String getPassword() {
        return password;
    }

    public String getDb() {
        return db;
    }

    public boolean isTruncateTable() {
        return truncateTable;
    }

    public int getRecordCount() {
        return recordCount;
    }

    public List<String> getIncludeTables() {
        return includeTables;
    }

    public List<String> getExcludeTables() {
        return excludeTables;
    }
}