/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.utils;

import io.gitee.jaemon.mocker.entity.MockDate;
import io.gitee.jaemon.mocker.entity.eunms.Gender;

import java.io.UnsupportedEncodingException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Random;

/**
 * mock工具类
 *
 * @author Jaemon
 * @since 1.0
 */
public final class RandomUtils {

    private RandomUtils() {
    }

    private static final char[] UPPER_CASE = "ABCDEFGHIJKLMNOPQRSTUVWXYZ".toCharArray();
    private static final char[] LOWER_CASE = "abcdefghijklmnopqrstuvwxyz".toCharArray();
    private static final char[] UPPER_LOWER_CASE = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz".toCharArray();
    private static final char[] NUM_CASE = "0123456789".toCharArray();
    private static final char[] CHAR_NUM_CASE = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789".toCharArray();
    private static final char[] LOWER_CHAR_NUM_CASE = "abcdefghijklmnopqrstuvwxyz0123456789".toCharArray();

    /** 年份界限值 */
    private static final int BOUNDARY_YEAR = 1700;
    /** 身份证最后一位计算比率 */
    private static final int[] RATIO = {7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2};
    /** 身份证最后一位 */
    private static final String[] LAST_NUM = {"1", "0", "X", "9", "8", "7", "6", "5", "4", "3", "2"};
    /** 手机号前三位 */
    private static final String[] TEL_FIRST_THREE_CHAR = {
            // 中国移动
            "134", "135", "136", "137", "138", "139", "150", "151", "152", "157", "158", "159", "182", "183", "184", "187", "188", "178", "147", "172", "198",
            // 中国联通
            "130", "131", "132", "145", "155", "156", "166", "171", "175", "176", "185", "186", "166",
            // 中国电信
            "133", "149", "153", "173", "177", "180", "181", "189", "199"
    };
    /** 邮箱后缀 */
    private static final String[] EMAIL_SUFFIX = {"@gmail.com", "@yahoo.com", "@msn.com", "@hotmail.com", "@aol.com", "@ask.com", "@live.com", "@qq.com", "@0355.net", "@163.com","@163.net", "@263.net", "@3721.net", "@yeah.net", "@googlemail.com", "@126.com", "@sina.com", "@sohu.com", "@yahoo.com.cn"};

    /** ip范围 */
    private static final int[][] range = {
            // 36.56.0.0-36.63.255.255
            {607649792, 608174079},
            // 61.232.0.0-61.237.255.255
            {1038614528, 1039007743},
            // 106.80.0.0-106.95.255.255
            {1783627776, 1784676351},
            // 121.76.0.0-121.77.255.255
            {2035023872, 2035154943},
            // 123.232.0.0-123.235.255.255
            {2078801920, 2079064063},
            // 139.196.0.0-139.215.255.255
            {-1950089216, -1948778497},
            // 171.8.0.0-171.15.255.255
            {-1425539072, -1425014785},
            // 182.80.0.0-182.92.255.255
            {-1236271104, -1235419137},
            // 210.25.0.0-210.47.255.255
            {-770113536, -768606209},
            // 222.16.0.0-222.95.255.255
            {-569376768, -564133889}
    };

    /** 连接符 */
    public static final String CONNECTOR = "-";
    public static final Random random = new Random();
    public static final DateTimeFormatter DATETIME_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");


    /**
     * 随机生成长度为n的字母字符串
     *
     * @param length
     *      字符串长度
     * @return
     *      字母字符串内容
     */
    public static String generateChars(int length) {
        return generate(UPPER_LOWER_CASE, length);
    }


    /**
     * 随机生成长度为n的数字字符串
     *
     * @param length
     *      字符串长度
     * @return
     *      数字字符串内容
     */
    public static String generateNums(int length) {
        return generate(NUM_CASE, length);
    }


    /**
     * 随机生成长度为n的字符串
     *
     * @param length
     *      字符串长度
     * @return
     *      字符串内容
     */
    public static String generateCharNums(int length) {
        return generate(CHAR_NUM_CASE, length);
    }


    /**
     * 随机生成日期字符串
     *
     * @return
     *      日期对象
     */
    public static MockDate date() {
        return date(LocalDate.now().getYear(), BOUNDARY_YEAR);
    }

    /**
     * 随机生成日期字符串
     *
     * @param upper
     *      上限年份, 1700-upper
     * @param lower
     *      下限, lower-now
     * @return
     *      日期对象
     */
    public static MockDate date(int upper, int lower) {
        upper = Math.max(upper, BOUNDARY_YEAR);
        lower = Math.max(lower, BOUNDARY_YEAR);
        int yearBound = upper + 1 - lower;
        int year = random.nextInt(yearBound) + lower;
        int month = random.nextInt(12) + 1;
        String monthVal = month < 10 ? "0" + month : String.valueOf(month);
        int dayBound = month == 2 ? (year % 4 == 0) ? 28 : 29 : 30;
        int day = random.nextInt(dayBound) + 1;
        String dayVal = day < 10 ? "0" + day : String.valueOf(day);

        return MockDate.instance(year, monthVal, dayVal);
    }

    /**
     * 随机生成身份证(1700~至今)
     *
     * @return
     *      身份证号
     */
    public static String idNo() {
        return idNo(LocalDate.now().getYear(), BOUNDARY_YEAR, Gender.RANDOM);
    }


    /**
     * 随机生成身份证
     *
     * @param upper
     *      上限年份
     * @param lower
     *      下限年份
     * @return
     *      身份证号
     */
    public static String idNo(int upper, int lower) {
        return idCard(upper, lower, Gender.RANDOM).idNo();
    }


    /**
     * 随机生成身份证(1700~至今)
     *
     * @return
     *      身份证号
     */
    public static String idNoMale() {
        return idNo(LocalDate.now().getYear(), BOUNDARY_YEAR, Gender.MALE);
    }


    /**
     * 随机生成身份证(1700~至今)
     *
     * @return
     *      身份证号
     */
    public static String idNoFemale() {
        return idNo(LocalDate.now().getYear(), BOUNDARY_YEAR, Gender.FEMALE);
    }


    /**
     * 随机生成身份证
     *
     * @param upper
     *      上限年份
     * @param lower
     *      下限年份
     * @param gender
     *      性别 {@link Gender}
     * @return
     *      身份证号
     */
    public static String idNo(int upper, int lower, Gender gender) {
        return idCard(upper, lower, gender).idNo();
    }


    /**
     * 随机生成身份证信息，带所属行政区(1700~至今)
     *
     * @return
     *      身份证信息
     */
    public static IdCard idCard() {
        return idCard(LocalDate.now().getYear(), BOUNDARY_YEAR, Gender.RANDOM);
    }

    /**
     * 随机生成男性身份证信息，带所属行政区(1700~至今)
     *
     * @return
     *      身份证信息
     */
    public static IdCard idCardMale() {
        return idCard(LocalDate.now().getYear(), BOUNDARY_YEAR, Gender.MALE);
    }

    /**
     * 随机生成女性身份证信息，带所属行政区(1700~至今)
     *
     * @return
     *      身份证信息
     */
    public static IdCard idCardFemale() {
        return idCard(LocalDate.now().getYear(), BOUNDARY_YEAR, Gender.FEMALE);
    }

    /**
     * 随机生成身份证信息，带所属行政区
     *
     * @param upper
     *      上限年份
     * @param lower
     *      下限年份
     * @param gender
     *      性别 {@link Gender}
     * @return
     *      身份证信息
     */
    public static IdCard idCard(int upper, int lower, Gender gender) {
        gender = gender == null ? Gender.RANDOM : gender;
        StringBuilder result = new StringBuilder(18);
        DomicilePlace domicilePlace = IdCardUtils.random();
        result.append(domicilePlace.code());
        result.append(date(upper, lower).toDate());
        result.append(generateNums(2));
        result.append(byteRandom(gender.value()));
        int sum = 0;
        for (int i = 0; i < result.length(); i++) {
            int num = Integer.parseInt(
                    result.substring(i, i + 1)
            );
            sum += num * RATIO[i];
        }
        int index = sum % 11;
        result.append(LAST_NUM[index]);
        return IdCard.idCard(result.toString(), domicilePlace.canton());
    }


    /**
     * 从int数组中随机获取元素
     *
     * @param ints
     *      数组内容
     * @return
     *      随机元素
     */
    public static int intRandom(int[] ints) {
        return ints[random.nextInt(ints.length)];
    }

    /**
     * 从byte数组中随机获取元素
     *
     * @param bytes
     *      数组内容
     * @return
     *      随机元素
     */
    public static byte byteRandom(byte[] bytes) {
        return bytes[random.nextInt(bytes.length)];
    }

    /**
     * 从字符串数组中随机获取元素
     *
     * @param stirngs
     *      数组内容
     * @return
     *      随机元素
     */
    public static String stringRandom(String[] stirngs) {
        return stirngs[random.nextInt(stirngs.length)];
    }

    /**
     * 随机生成长度为length的字符串
     *
     * @param source
     *          数据源
     * @param length
     *          字符串长度长度
     * @return
     *          字符串内容
     */
    private static String generate(char[] source, int length) {
        StringBuilder result = new StringBuilder(length);

        for (int i = 0; i < length; i++) {
            int index = random.nextInt(source.length);
            result.append(source[index]);
        }

        return result.toString();
    }


    /**
     * 随机生成中文名
     *
     * @return
     *      中文姓名
     */
    public static String randomCNName(){
        StringBuilder name = new StringBuilder();
        String firstName = stringRandom(FirstName.fistNames);
        name.append(firstName);
        int lastNameCnt = random.nextInt(2) + 1;
        for (int i = 0; i < lastNameCnt; i++) {
            name.append(chinese());
        }

        return name.toString();
    }

    /**
     * 随机生成邮箱
     *
     * @return
     *      邮箱地址
     */
    public static String randomEmail() {
        StringBuilder email = new StringBuilder();
        email.append(generateChars(1));
        email.append(
                generate(
                        LOWER_CHAR_NUM_CASE, random.nextInt(16) + 2
                )
        );
        email.append(stringRandom(EMAIL_SUFFIX));
        return email.toString();
    }

    /**
     * 随机生成QQ邮箱
     *
     * @return
     *      QQ邮箱地址
     */
    public static String randomQQEmail() {
        StringBuilder email = new StringBuilder();
        email.append(random.nextInt(9) + 1);
        email.append(generateNums(random.nextInt(3) + 6));
        email.append(EMAIL_SUFFIX[7]);
        return email.toString();
    }

    /**
     * 随机生成手机号
     *
     * @return
     *      手机号
     */
    public static String randomTel() {
        StringBuilder tel = new StringBuilder();
        tel.append(stringRandom(TEL_FIRST_THREE_CHAR));
        tel.append(generateNums(8));
        return tel.toString();
    }

    /**
     * 随机生成IP地址
     *
     * @return
     *      IP地址
     */
    public static String randomIp() {
        int index = random.nextInt(10);
        return num2ip(
                range[index][0] + random.nextInt(range[index][1] - range[index][0])
        );
    }

    /**
     * 生成汉字
     *
     * <pre>
     *     GBK 编码表： https://www.qqxiuzi.cn/zh/hanzi-gbk-bianma.php
     *     查看方式如: 中(D6D0) 左上角(D6)+纵坐标(D)+横坐标(0)
     *
     *     high position:   B0-D7(176-215)
     *     low position:    A1-FE(161-254)
     * </pre>
     *
     * @return
     *      汉字
     */
    public static String chinese() {
        byte[] bytes = new byte[2];
        Random random = new Random();
        int highPosition = random.nextInt(39) + 176;
        int lowPosition = random.nextInt(93) + 161;
        bytes[0] = (byte) (highPosition > 127 ? highPosition - 256 : highPosition);
        bytes[1] = (byte) (lowPosition > 127 ? lowPosition - 256 : lowPosition);
        try {
            return new String(bytes, "GBK");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 将十进制转换成IP地址
     *
     * @param ip
     *      ip数值
     * @return
     *      ip地址
     */
    private static String num2ip(int ip) {
        int[] b = new int[4];
        b[0] = (ip >> 24) & 0xff;
        b[1] = (ip >> 16) & 0xff;
        b[2] = (ip >> 8) & 0xff;
        b[3] = ip & 0xff;
        return b[0] + "." + b[1] + "." + b[2] + "." + b[3];
    }


    /**
     * 生成汉字
     *
     * @return
     *      result
     */
    private static char randomCNChar(){
        return (char) (0x4e00 + (int) (random.nextDouble() * (0x9fa5 - 0x4e00 + 1)));
    }

}