/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.utils;

import java.util.ArrayList;
import java.util.List;

/**
 * IdCardUtils
 *
 * @author Jaemon
 * @version 1.0
 */
public final class IdCardUtils {

    private IdCardUtils() {
    }

    /**
     * 随机生成行政区码和信息
     *
     * @return
     *      {@link DomicilePlace}
     */
    public static DomicilePlace random() {
        StringBuilder key = new StringBuilder();
        key.append(IdCardRecognition.PROVINCE_KEY);
        Canton canton = new Canton();
        String code;

        String[] provinces = IdCardRecognition.map.get(key.toString());
        int pIdx = RandomUtils.random.nextInt(provinces.length);

        String[] provinceArr = provinces[pIdx].split(IdCardRecognition.VALUE_SEPERATOR);
        canton.setProvince(provinceArr[0]);
        code = provinceArr[1];
        key.append(IdCardRecognition.KEY_SEPERATOR).append(pIdx);

        String[] citys = IdCardRecognition.map.get(key.toString());
        if (citys != null) {
            int cIdx = RandomUtils.random.nextInt(citys.length);
            String[] cityArr = citys[cIdx].split(IdCardRecognition.VALUE_SEPERATOR);
            canton.setCity(cityArr[0]);
            code = cityArr[1];
            key.append(IdCardRecognition.KEY_SEPERATOR).append(cIdx);
        }

        String[] regions = IdCardRecognition.map.get(key.toString());
        if (regions != null) {
            int rIdx = RandomUtils.random.nextInt(regions.length);
            String[] regionArr = regions[rIdx].split(IdCardRecognition.VALUE_SEPERATOR);
            canton.setRegion(regionArr[0]);
            code = regionArr[1];
        }

        return DomicilePlace.domicilePlace(code, canton);
    }

    /**
     * 根据行政区信息获取行政区码
     *
     * @param canton
     *          行政区信息
     * @return
     *          行政区码
     */
    public static String code(Canton canton) {
        String code = null;
        StringBuilder key = new StringBuilder();
        key.append(IdCardRecognition.PROVINCE_KEY);
        String[] provinces = IdCardRecognition.map.get(key.toString());
        int pIdx = -1, cIdx = -1;
        OK:
        for (int i = 0; i < provinces.length; i++) {
            String[] provinceArr = provinces[i].split(IdCardRecognition.VALUE_SEPERATOR);
            if (provinceArr[0].equals(canton.getProvince())) {
                pIdx = i;
                code = provinceArr[1];
                break OK;
            }
        }

        if (pIdx != -1) {
            key.append(IdCardRecognition.KEY_SEPERATOR).append(pIdx);
            String[] citys = IdCardRecognition.map.get(key.toString());
            OK:
            for (int i = 0; i < citys.length; i++) {
                String[] cityArr = citys[i].split(IdCardRecognition.VALUE_SEPERATOR);
                if (cityArr[0].equals(canton.getCity())) {
                    cIdx = i;
                    code = cityArr[1];
                    break OK;
                }
            }
        }

        if (cIdx != -1) {
            key.append(IdCardRecognition.KEY_SEPERATOR).append(cIdx);
            String[] regions = IdCardRecognition.map.get(key.toString());
            OK:
            for (int i = 0; i < regions.length; i++) {
                String[] regionArr = regions[i].split(IdCardRecognition.VALUE_SEPERATOR);
                if (regionArr[0].equals(canton.getRegion())) {
                    code = regionArr[1];
                    break OK;
                }
            }
        }

        return code;
    }

    /**
     * 根据行政区码获取行政区信息
     *
     * @param code
     *          行政区码
     * @return
     *          行政区信息
     */
    public static Canton canton(String code) {
        Canton canton = new Canton();
        if (StringUtils.isEmpty(code) || code.length() != 6) {
            StringUtils.print("Illegal canton code %s.", code);
            return canton;
        }

        StringBuilder key = new StringBuilder();
        key.append(IdCardRecognition.PROVINCE_KEY);
        String[] provinces = IdCardRecognition.map.get(key.toString());
        List<String> codes = codes(code);

        OK:
        for (int i = 0; i < provinces.length; i++) {
            String[] provinceArr = provinces[i].split(IdCardRecognition.VALUE_SEPERATOR);
            if (codes.contains(provinceArr[1])) {
                canton.setProvince(provinceArr[0]);
                key.append(IdCardRecognition.KEY_SEPERATOR).append(i);
                break OK;
            }
        }

        String[] citys = IdCardRecognition.map.get(key.toString());
        if (citys != null) {
            OK:
            for (int i = 0; i < citys.length; i++) {
                String[] cityArr = citys[i].split(IdCardRecognition.VALUE_SEPERATOR);
                if (codes.contains(cityArr[1])) {
                    canton.setCity(cityArr[0]);
                    key.append(IdCardRecognition.KEY_SEPERATOR).append(i);
                    break OK;
                }
            }
        }

        String[] regions = IdCardRecognition.map.get(key.toString());
        if (regions != null) {
            OK:
            for (int i = 0; i < regions.length; i++) {
                String[] regionArr = regions[i].split(IdCardRecognition.VALUE_SEPERATOR);
                if (codes.contains(regionArr[1])) {
                    canton.setRegion(regionArr[0]);
                    break OK;
                }
            }
        }

        return canton;
    }

    private static List<String> codes(String code) {
        List<String> codes = new ArrayList<>();
        codes.add(code.substring(0, 2) + "0000");
        codes.add(code.substring(0, 4) + "00");
        codes.add(code);
        return codes;
    }

}