/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.template.clazz;

import io.gitee.jaemon.mocker.entity.TemplateColumn;
import io.gitee.jaemon.mocker.template.AbstractClassGenerator;
import io.gitee.jaemon.mocker.template.ClassGeneratorContext;
import io.gitee.jaemon.mocker.common.Constants;
import io.gitee.jaemon.mocker.entity.eunms.FileType;
import io.gitee.jaemon.mocker.entity.eunms.TemplateType;
import io.gitee.jaemon.mocker.template.DataBaseDataType;

import java.util.List;

/**
 * 拼接式Mapper映射文件生成器
 *
 * @author Jaemon
 * @since 1.0
 */
public class MapperGenerator extends AbstractClassGenerator {

    @Override
    public String frame(ClassGeneratorContext context) {
        List<TemplateColumn> columns = context.getColumns();
        String upperCamel = context.getUpperCamel();
        StringBuilder head = new StringBuilder();
        head.append("\t<sql id=\""+ upperCamel +"Columns\">\n\t\t");
        for (TemplateColumn column : columns) {
            head.append(column.getColumnName()).append(", ");
        }
        if (!columns.isEmpty()) {
            head.deleteCharAt(head.length() - 2);
        }
        head.append("\n\t</sql>\n\n");


        return "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n" +
                "<!DOCTYPE mapper PUBLIC \"-//mybatis.org//DTD Mapper 3.0//EN\" \"http://mybatis.org/dtd/mybatis-3-mapper.dtd\">\n" +
                "<mapper namespace=\"" + context.getBasePackage() + "." + TemplateType.DAO.pkgName() + "." + context.daoUc() + "\">\n" +
                head + "%s" +
                "</mapper>";
    }

    @Override
    protected String queryList(ClassGeneratorContext context) {
        StringBuilder sql = new StringBuilder();
        sql.append("\t\t" +
                "select <include refid=\""+ context.getUpperCamel() +"Columns\" />\n\t\t" +
                "from " + context.getTableName() + "\n\t\t" +
                "<where>\n");

        List<TemplateColumn> columns = context.getColumns();
        for (int i = 0, size = columns.size(); i < size; i++) {
            TemplateColumn column = columns.get(i);
            String columnName = column.getColumnName();
            String lowerCamelColumnName = Constants.LOWER_UNDERSCORE_CAMEL_CONVERT.convert(columnName);
            DataBaseDataType dataBaseDataType = DataBaseDataType.matching(column.getColumnType());
            String condition;
            if (dataBaseDataType.packType().equals(String.class.getSimpleName())) {
                condition = "and "+ columnName +" LIKE CONCAT('%', #{"+ lowerCamelColumnName +"}, '%')";
            } else {
                condition = "and "+ columnName +" = #{"+ lowerCamelColumnName +"}";
            }
            sql.append("\t\t\t")
                    .append("<if test=\""+ lowerCamelColumnName +" != null and "+ lowerCamelColumnName +" != ''\">\n\t\t\t\t")
                    .append(condition).append("\n\t\t\t</if>\n");
        }
        sql.append("\t\t</where>\n");

        String methodName = Thread.currentThread().getStackTrace()[1].getMethodName();
        return "<select id=\""+ methodName +"\" resultType=\"" + context.getBasePackage() + "." + TemplateType.ENTITY.pkgName() + "." + context.getUpperCamel() + "\">\n" +
                sql +
                "\t</select>";
    }

    @Override
    protected String queryByPrimaryKey(ClassGeneratorContext context) {
        StringBuilder sql = new StringBuilder();
        sql.append("\t\t" +
                "select <include refid=\""+ context.getUpperCamel() +"Columns\" />\n\t\t" +
                "from " + context.getTableName() + "\n");

        primaryKeysCondition(sql, context);

        String methodName = Thread.currentThread().getStackTrace()[1].getMethodName();
        return "<select id=\""+ methodName +"\" resultType=\"" + context.getBasePackage() + "." + TemplateType.ENTITY.pkgName() + "." + context.getUpperCamel() + "\">\n" +
                sql +
                "\n\t</select>";
    }

    @Override
    protected String insert(ClassGeneratorContext context) {
        StringBuilder sql = new StringBuilder();
        sql.append("insert into ").append(context.getTableName()).append("( <include refid=\""+ context.getUpperCamel() +"Columns\" /> )\n\t\tvalues(\n\t\t\t");
        List<TemplateColumn> columns = context.getColumns();
        for (int i = 0, size = columns.size(); i < size; i++) {
            TemplateColumn column = columns.get(i);
            sql.append("#{"+ Constants.LOWER_UNDERSCORE_CAMEL_CONVERT.convert(column.getColumnName()) +"}").append((i == size - 1) ? "" : ", ");
        }
        sql.append("\n\t\t)");

        String methodName = Thread.currentThread().getStackTrace()[1].getMethodName();
        return "<insert id=\""+ methodName +"\" parameterType=\"" + context.getBasePackage() + "." + TemplateType.ENTITY.pkgName() + "." + context.getUpperCamel() + "\">\n\t\t" +
                sql +
                "\n\t</insert>";
    }

    @Override
    protected String batchInsert(ClassGeneratorContext context) {
        StringBuilder sql = new StringBuilder();
        sql.append("insert into ").append(context.getTableName()).append("( <include refid=\""+ context.getUpperCamel() +"Columns\" /> )\n\t\tvalues\n");
        sql.append("\t\t<foreach collection=\""+ context.getLowerCamel() +"s\" item=\"item\" index=\"index\" separator=\",\">\n");
        sql.append("\t\t\t(\n\t\t\t\t");

        List<TemplateColumn> columns = context.getColumns();
        for (int i = 0, size = columns.size(); i < size; i++) {
            TemplateColumn column = columns.get(i);
            sql.append("#{item."+ Constants.LOWER_UNDERSCORE_CAMEL_CONVERT.convert(column.getColumnName()) +"}").append((i == size - 1) ? "" : ", ");
        }
        sql.append("\n\t\t\t)\n\t\t");
        sql.append("</foreach>");

        String methodName = Thread.currentThread().getStackTrace()[1].getMethodName();
        return "<insert id=\""+ methodName +"\">\n\t\t" +
                sql +
                "\n\t</insert>";
    }

    @Override
    protected String update(ClassGeneratorContext context) {
        StringBuilder sql = new StringBuilder();
        sql.append("update ").append(context.getTableName()).append("\n\t\t");
        sql.append("<trim prefix=\"set\" suffixOverrides=\",\">");

        List<TemplateColumn> columns = context.getColumns();
        for (int i = 0, size = columns.size(); i < size; i++) {
            TemplateColumn column = columns.get(i);
            String columnName = column.getColumnName();
            String lowerCamelColumnName = Constants.LOWER_UNDERSCORE_CAMEL_CONVERT.convert(columnName);
            sql.append("\n\t\t\t")
                    .append("<if test=\""+ lowerCamelColumnName +" != null \">\n\t\t\t\t")
                    .append(""+ columnName +" = #{"+ lowerCamelColumnName +"},")
                    .append("\n\t\t\t</if>");
        }
        sql.append("\n\t\t</trim>\n");
        primaryKeysCondition(sql, context);

        String methodName = Thread.currentThread().getStackTrace()[1].getMethodName();
        return "<update id=\""+ methodName +"\" parameterType=\"" + context.getBasePackage() + "." + TemplateType.ENTITY.pkgName() + "." + context.getUpperCamel() + "\">\n\t\t" +
                sql +
                "\n\t</update>";
    }

    @Override
    protected String deleteByPrimaryKey(ClassGeneratorContext context) {
        StringBuilder sql = new StringBuilder();
        sql.append("delete from ").append(context.getTableName()).append("\n");
        primaryKeysCondition(sql, context);

        String methodName = Thread.currentThread().getStackTrace()[1].getMethodName();
        return "<delete id=\""+ methodName +"\">\n\t\t" +
                sql +
                "\n\t</delete>";
    }

    @Override
    public boolean notes() {
        return false;
    }

    @Override
    public FileType suffix() {
        return FileType.XML;
    }

    private void primaryKeysCondition(StringBuilder sql, ClassGeneratorContext context) {
        for (int i = 0, size = context.getPrimaryKeys().size(); i < size; i++) {
            ClassGeneratorContext.PrimaryKey primaryKey = context.getPrimaryKeys().get(i);
            sql.
                    append((i == 0) ? "\t\twhere " : "")
                    .append(primaryKey.getName()).append(" = ")
                    .append("#{"+ Constants.LOWER_UNDERSCORE_CAMEL_CONVERT.convert(primaryKey.getName()) +"}")
                    .append((i == size - 1) ? "" : " and ");
        }
    }
}