/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.template;

import io.gitee.jaemon.mocker.exception.MockException;
import io.gitee.jaemon.mocker.entity.TemplateColumn;
import io.gitee.jaemon.mocker.common.Constants;
import io.gitee.jaemon.mocker.core.TemplateContext;
import io.gitee.jaemon.mocker.entity.eunms.FileType;
import io.gitee.jaemon.mocker.entity.eunms.TemplateType;
import io.gitee.jaemon.mocker.utils.FileUtils;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static io.gitee.jaemon.mocker.utils.StringUtils.print;

/**
 * 模板文件生成器
 *
 * @author Jaemon
 * @since 1.0
 */
public abstract class AbstractTemplateGenerator {
    private static final String USER_DIR = System.getProperty("user.dir") + File.separator;

    /**
     * 待生成模板的表集合
     *
     * @param tables
     *      指定表集合
     * @return
     *      表集合
     */
    abstract Map<String, List<TemplateColumn>> tables(List<String> tables);

    /**
     * 生成模板
     *
     * @param template
     *      模板配置
     */
    public final void generate(TemplateContext template) throws MockException {
        String javaPath, resourcesPath;
        String projectBaseDir = USER_DIR + "src/main";

        if (new File(projectBaseDir).exists()) {
            javaPath = projectBaseDir + File.separator + "java" +
                    File.separator + template.getBasePackage() + File.separator;
            resourcesPath = projectBaseDir +
                    File.separator + "resources" + File.separator;
        } else {
            javaPath = USER_DIR;
            resourcesPath = USER_DIR;
        }

        Map<String, List<TemplateColumn>> tables = tables(template.tables());

        if (tables.isEmpty()) {
            print("待生成模板的表个数为空!!!");
            return;
        }

        Set<String> tableNames = tables.keySet();
        print(". 本次任务待生成模板的表信息%s.", tableNames);

        for (Map.Entry<String, List<TemplateColumn>> table : tables.entrySet()) {
            String tableName = table.getKey();
            print("├── table=%s 开始模板生成啦~~", tableName);
            List<TemplateColumn> columns = table.getValue();

            ClassGeneratorContext context = new ClassGeneratorContext();
            context.setTableName(tableName);
            String name = tableName;
            for (String prefix : template.prefixs()) {
                if (name.startsWith(prefix) || name.endsWith(prefix)) {
                    name = name.replaceAll(prefix, "");
                }
            }
            context.setUpperCamel(Constants.UPPER_UNDERSCORE_CAMEL_CONVERT.convert(name));
            context.setLowerCamel(Constants.LOWER_UNDERSCORE_CAMEL_CONVERT.convert(name));
            List<ClassGeneratorContext.PrimaryKey> primaryKeys = columns.stream().
                    filter(TemplateColumn::isPrimaryKey).
                    map(e ->
                            new ClassGeneratorContext.PrimaryKey(
                                    e.getColumnType(), e.getColumnName(), e.getColumnComment()
                            )
                    ).
                    collect(Collectors.toList());
            context.setPrimaryKeys(primaryKeys);
            context.setTemplate(template);
            context.setColumns(columns);

            for (TemplateType templateType : TemplateType.values()) {
                print("│   ├── table=%s 准备开始生成%s模板.", tableName, templateType.name());

                if (templateType.classGenerator() == null) {
                    continue;
                }
                ClassGenerator classGenerator = TemplateGeneratorFactory.get(
                        templateType.classGenerator().getClass().getName()
                );
                if (classGenerator == null) {
                    continue;
                }
                String generator = classGenerator.generator(context);
                try {
                    FileType templateSuffix = classGenerator.suffix();
                    String generatePath = templateSuffix == FileType.JAVA ?
                            javaPath + templateType.pkgName() + File.separator : resourcesPath;
                    generatePath = generatePath + name(templateType, context);
                    File generatorFile = new File(
                            generatePath.replaceAll("\\.", "/") + templateSuffix.type()
                    );
                    FileUtils.writeStringToFile(generatorFile, generator);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            print("├── table=%s 模板生成完成!", tableName);
        }

        print(". 本次任务共完成%s张表的模板生成.", tables.size());
    }

    /**
     * 模板名称
     *
     * @param templateType
     *      模板类型
     * @param context
     *      上下文对象
     * @return
     *      模板名称
     */
    private static String name(TemplateType templateType, ClassGeneratorContext context) {
        if (templateType == TemplateType.SERVICE) {
            return context.serviceUc();
        } else if (templateType == TemplateType.SERVICEIMPL) {
            return context.serviceImplUc();
        } else if (templateType == TemplateType.DAO) {
            return context.daoUc();
        } else if (templateType == TemplateType.MAPPER) {
            return context.mapperUc();
        }

        return context.getUpperCamel();
    }

}