/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.template;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * AbstractClassGenerator
 *
 * @author Jaemon
 * @since 1.0
 */
public abstract class AbstractClassGenerator extends AbstractClassMethods {

    /**
     * 类内容框架
     *
     * @param context
     *      上下文信息
     * @return
     *      类内容框架字符串
     */
    public abstract String frame(ClassGeneratorContext context);

    @Override
    public final String generator(ClassGeneratorContext context) {
        return String.format(frame(context), body(context));
    }

    /**
     * 类方法体内容
     *
     * @param context
     *      上下文信息
     * @return
     *      类内容体字符串
     */
    protected String body(ClassGeneratorContext context) {
        StringBuilder body = new StringBuilder();
        body.append("\n");
        Method[] declaredMethods = AbstractClassMethods.class.getDeclaredMethods();
        List<Method> methods = Stream.of(declaredMethods).filter(
                e -> e.getModifiers() == Modifier.PROTECTED + Modifier.ABSTRACT &&
                        e.isAnnotationPresent(MethodGenerator.class) &&
                        e.getReturnType().equals(String.class)
        ).sorted(Comparator.comparingInt(x -> x.getAnnotation(MethodGenerator.class).order())).collect(Collectors.toList());

        for (int i = 0, size = methods.size(); i < size; i++) {
            Method declaredMethod = methods.get(i);

            String notes = "";
            if (notes()) {
                notes = methodNoteGenerator(
                        context.getPrimaryKeys(),
                        declaredMethod.getAnnotation(MethodGenerator.class)
                );
            }

            String override = "";
            if (override()) {
                override = "\t@Override\n";
            }

            try {
                Object result = declaredMethod.invoke(this, context);
                if (String.class.isInstance(result)) {
                    if (result != null && !"".equals(result)) {
                        body.append(notes).append(override).append("\t").append(result).
                                append((i  != size - 1) ? "\n\n" : "\n");
                    }
                }
            } catch (IllegalAccessException | InvocationTargetException e) {
                e.printStackTrace();
            }
        }
        return body.toString();
    };


    /**
     * 主键形参
     *
     * @param primaryKeys
     *      主键信息
     * @return
     *      主键形参字符串
     */
    protected final StringBuilder primaryKeysGenerator(List<ClassGeneratorContext.PrimaryKey> primaryKeys) {
        return primaryKeysGenerator(primaryKeys, false);
    }

    /**
     * 主键形参
     *
     * @param primaryKeys
     *      主键信息
     * @param paramAnnotation
     *      是否需要带Param注解
     * @return
     *      主键形参字符串
     */
    protected final StringBuilder primaryKeysGenerator(
            List<ClassGeneratorContext.PrimaryKey> primaryKeys, boolean paramAnnotation
    ) {
        StringBuilder params = new StringBuilder();
        if (primaryKeys == null) {
            return params;
        }

        for (int i = 0, size = primaryKeys.size(); i < size; i++) {
            ClassGeneratorContext.PrimaryKey primaryKey = primaryKeys.get(i);
            String param = paramAnnotation ? String.format("@Param(\"%s\") ", primaryKey.getName()) : "";
            DataBaseDataType dataBaseDataType = DataBaseDataType.matching(primaryKey.getType());
            params.
                    append(param).
                    append(dataBaseDataType.packType()).
                    append(" ").append(primaryKey.getName()).
                    append((i == size - 1) ? "" : ", ");
        }

        return params;
    }

    /**
     * 方法注释生成
     *
     * @param primaryKeys
     *      主键信息
     * @param methodGenerator
     *      methodGenerator {@link MethodGenerator}
     * @return
     *      方法注释内容
     */
    private String methodNoteGenerator(
            List<ClassGeneratorContext.PrimaryKey> primaryKeys, MethodGenerator methodGenerator
    ) {
        StringBuilder paramsNote = new StringBuilder();
        StringBuilder returnNote = new StringBuilder();

        String paramNoteTemplate = " * @param %s\n\t" +
                " *\t\t%s\n\t";
        String returnNoteTemplate = " * @return \n\t" +
                " *\t\t%s\n\t";

        if (methodGenerator.primaryKeyParam()) {
            for (ClassGeneratorContext.PrimaryKey primaryKey : primaryKeys) {
                paramsNote.append(String.format(paramNoteTemplate, primaryKey.getName(), primaryKey.getDesc()));
            }
        }

        for (MethodGenerator.ParamsNote pn : methodGenerator.paramsNote()) {
            paramsNote.append(String.format(paramNoteTemplate, pn.name(), pn.desc()));
        }

        String result = methodGenerator.result();
        if (result != null && !"".equals(result)) {
            returnNote.append(String.format(returnNoteTemplate, result));
        }

        return "\t/**\n\t" +
                " * " + methodGenerator.value() + "\n\t" +
                " * \n\t" +
                paramsNote.toString() +
                returnNote.toString() +
                " */\n";
    }

}