/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.ftl;

import io.gitee.jaemon.mocker.common.Constants;
import io.gitee.jaemon.mocker.template.DataBaseDataType;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 表定义
 *
 * @author Jaemon
 * @since 1.0
 */
public class TableDefinition {
    /** 表名 */
    private String tableName;
    /** 实体名 */
    private String modelName;
    /** 驼峰实体名 */
    private String camelModelName;
    /** 字段集 */
    private List<ColumnDefinition> columns;

    public TableDefinition(String tableName) {
        this(tableName, new ArrayList<>());
    }

    public TableDefinition(String tableName, List<String> prefixs) {
        this.tableName = tableName;
        String name = tableName;
        for (String prefix : prefixs) {
            if (name.startsWith(prefix) || name.endsWith(prefix)) {
                name = name.replaceAll(prefix, "");
            }
        }
        this.modelName = Constants.UPPER_UNDERSCORE_CAMEL_CONVERT.convert(name);
        this.camelModelName = Constants.LOWER_UNDERSCORE_CAMEL_CONVERT.convert(name);
    }

    public String getTableName() {
        return tableName;
    }

    public String getModelName() {
        return modelName;
    }

    public String getCamelModelName() {
        return camelModelName;
    }

    public List<ColumnDefinition> getColumns() {
        return columns;
    }

    public void setColumns(List<ColumnDefinition> columns) {
        this.columns = columns;
    }

    public List<ColumnDefinition> getPrimaryKeys() {
        return columns.stream().filter(e -> e.isPrimaryKey()).collect(Collectors.toList());
    }

    public boolean getHasDate() {
        return columns.stream().filter(e -> e.getDataType().classType() == Date.class).count() != 0;
    }

    public boolean getHasBigDecimal() {
        return columns.stream().filter(e -> e.getDataType().classType() == BigDecimal.class).count() != 0;
    }

    /**
     * 字段定义
     */
    public static class ColumnDefinition {
        /** 数据库字段名 */
        private String columnName;
        /** 字段类型 */
        private String columnType;
        /** 字段注释 */
        private String columnComment;
        /** 是否可为空 */
        private boolean nullable;
        /** 是否为主键 */
        private boolean primaryKey;
        /** 实体属性名 */
        private String propertyName;
        /** 属性java类型 */
        private String javaType;
        /** 数据类型 */
        private DataBaseDataType dataType;

        public ColumnDefinition(String columnName, String columnType, String columnComment, boolean nullable, boolean primaryKey) {
            this.columnName = columnName;
            this.columnType = columnType;
            this.columnComment = columnComment;
            this.nullable = nullable;
            this.primaryKey = primaryKey;
            this.propertyName = Constants.LOWER_UNDERSCORE_CAMEL_CONVERT.convert(columnName);
            DataBaseDataType dataBaseDataType = DataBaseDataType.matching(columnType);
            this.dataType = dataBaseDataType;
            this.javaType = nullable ? dataBaseDataType.packType() : dataBaseDataType.baseType();
        }

        public String getColumnName() {
            return columnName;
        }

        public String getColumnType() {
            return columnType;
        }

        public String getColumnComment() {
            return columnComment;
        }

        public boolean isNullable() {
            return nullable;
        }

        public boolean isPrimaryKey() {
            return primaryKey;
        }

        public String getPropertyName() {
            return propertyName;
        }

        public String getJavaType() {
            return javaType;
        }

        public DataBaseDataType getDataType() {
            return dataType;
        }
    }

}