/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.ftl;

import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 方法定义
 *
 * @author Jaemon
 * @version 1.0
 */
public class MethodDefinition {
    /**
     * 方法修饰
     *
     * @see Modifier
     * */
    private int methodModifier = Modifier.PUBLIC;
    /**
     * 方法返回类型, 默认返回空
     */
    public ReturnType returnType = new MethodDefinition.ReturnType(Void.TYPE, null);
    /**
     * 方法名
     */
    private String methodName;
    /**
     * 方法参数列表
     */
    private List<Param> methodParams;
    /**
     * 参数是否需带主键(和methodParams互斥), 默认false
     */
    private boolean primaryKey;
    /**
     * 方法注释
     */
    private String methodComments;

    public MethodDefinition(String methodName, String methodComments) {
        this.methodName = methodName;
        this.methodComments = methodComments;
    }

    public MethodDefinition(MethodDefinition.ReturnType returnType, String methodName, String methodComments) {
        this(methodName, methodComments);
        this.returnType = returnType;
    }

    public String getMethodModifier() {
        return Modifier.toString(methodModifier);
    }

    public void setMethodModifier(int methodModifier) {
        this.methodModifier = methodModifier;
    }

    public String getReturnType() {
        return returnType.toString();
    }

    public void setReturnType(MethodDefinition.ReturnType returnType) {
        this.returnType = returnType;
    }

    public String getMethodName() {
        return methodName;
    }

    public void setMethodName(String methodName) {
        this.methodName = methodName;
    }

    public List<MethodDefinition.Param> getMethodParams() {
        return methodParams;
    }

    public void setMethodParams(List<MethodDefinition.Param> methodParams) {
        this.methodParams = methodParams;
    }

    public boolean isPrimaryKey() {
        return primaryKey;
    }

    public void setPrimaryKey(boolean primaryKey) {
        this.primaryKey = primaryKey;
    }

    public String getMethodComments() {
        return methodComments;
    }

    public void setMethodComments(String methodComments) {
        this.methodComments = methodComments;
    }

    /**
     * 方法参数
     */
    public static class Param {
        /**
         * 参数类型
         */
        private Class paramType;
        /**
         * 参数名称
         */
        private String paramName;
        /** 参数注释 */
        private String paramComments;
        /**
         * 如果参数类型带泛型, 对应的泛型类型, 否则List默认{@code List<ModelName>}, Map默认{@code Map<String, ModelName>}
         */
        private List<Class<?>> genericTypes;
        private String modelName;

        public Param(Class paramType, String paramName, String paramComments, String modelName) {
            this.paramType = paramType;
            this.paramName = paramName;
            this.paramComments = paramComments;
            this.modelName = modelName;
        }

        public Param(Class paramType, String paramName, String paramComments, List<Class<?>> genericTypes, String modelName) {
            this(paramType, paramName, paramComments, modelName);
            this.genericTypes = genericTypes;
        }

        public String getParamType() {
            return generic(paramType, genericTypes, modelName);
        }

        public void setParamType(Class paramType) {
            this.paramType = paramType;
        }

        public String getParamName() {
            return paramName;
        }

        public void setParamName(String paramName) {
            this.paramName = paramName;
        }

        public String getParamComments() {
            return paramComments;
        }

        public void setParamComments(String paramComments) {
            this.paramComments = paramComments;
        }

        public List<Class<?>> getGenericTypes() {
            return genericTypes;
        }

        public void setGenericTypes(List<Class<?>> genericTypes) {
            this.genericTypes = genericTypes;
        }
    }

    /**
     * 方法返回类型
     */
    public static class ReturnType {
        /**
         * 返回类型, 默认为空
         */
        private Class returnType;
        /**
         * 如果参数类型带泛型, 对应的泛型类型, 否则List默认{@code List<ModelName>}, Map默认{@code Map<String, ModelName>}
         */
        private List<Class<?>> genericTypes;
        private String modelName;

        public ReturnType(Class returnType, String modelName) {
            this.returnType = returnType;
            this.modelName = modelName;
        }

        public ReturnType(Class returnType, List<Class<?>> genericTypes, String modelName) {
            this(returnType, modelName);
            this.genericTypes = genericTypes;
        }

        public Class getReturnType() {
            return returnType;
        }

        public void setReturnType(Class returnType) {
            this.returnType = returnType;
        }

        public List<Class<?>> getGenericTypes() {
            return genericTypes;
        }

        public void setGenericTypes(List<Class<?>> genericTypes) {
            this.genericTypes = genericTypes;
        }

        @Override
        public String toString() {
            return generic(returnType, genericTypes, modelName);
        }
    }

    private static String generic(Class classTypeType, List<Class<?>> genericTypes, String modelName) {
        if (classTypeType == MockModel.class) {
            return modelName;
        }

        if (Collection.class.isAssignableFrom(classTypeType)) {
            return String.format("%s<%s>",
                    classTypeType.getSimpleName(),
                    genericTypes == null || genericTypes.isEmpty() ?
                            modelName :
                            (genericTypes.get(0) == MockModel.class) ?
                                    modelName : genericTypes.get(0).getSimpleName()
            );
        }

        if (Map.class.isAssignableFrom(classTypeType)) {
            return String.format("%s<%s, %s>",
                    classTypeType.getSimpleName(),
                    genericTypes == null || genericTypes.isEmpty() ?
                            String.class.getSimpleName() :
                            (genericTypes.get(0) == MockModel.class) ?
                                    modelName :
                                    genericTypes.get(0).getSimpleName(),
                    genericTypes == null || genericTypes.size() < 2 ?
                            modelName :
                            (genericTypes.get(1) == MockModel.class) ?
                                    modelName : genericTypes.get(1).getSimpleName()
            );
        }
        return classTypeType.getSimpleName();
    }

}