/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.ftl;

import io.gitee.jaemon.mocker.exception.MockException;
import io.gitee.jaemon.mocker.core.TemplateContext;
import io.gitee.jaemon.mocker.utils.FreemarkerUtils;
import io.gitee.jaemon.mocker.core.SqlExecutor;
import io.gitee.jaemon.mocker.entity.TemplateColumn;
import io.gitee.jaemon.mocker.entity.eunms.FileType;

import java.io.*;
import java.util.*;
import java.util.stream.Collectors;

import static io.gitee.jaemon.mocker.utils.StringUtils.print;

/**
 * Freemark模板生成
 *
 * @author Jaemon
 * @since 1.0
 */
public class FreemarkGenerator extends TemplateGenerator<TableDefinition> {
    private Map<GenerateType, String> generateTypeMap;
    private Map<String, String> packageMap;

    public FreemarkGenerator(TemplateContext templateContext) {
        super(templateContext);
        generateTypeMap = new HashMap<>();
        generateTypeMap.put(GenerateType.ENTITY, templateContext.getEntity());
        generateTypeMap.put(GenerateType.SERVICE, templateContext.getService());
        generateTypeMap.put(GenerateType.SERVICEIMPL, templateContext.getServiceImpl());
        generateTypeMap.put(GenerateType.DAO, templateContext.getDao());
        generateTypeMap.put(GenerateType.MAPPER, templateContext.getMapper());

        packageMap = new HashMap<>();
        for (GenerateType generateType : GenerateType.values()) {
            packageMap.put(generateType.name().toLowerCase(), generateType.pkgName());
        }
    }

    @Override
    protected Map<String, TableDefinition> tables() {
        List<TemplateColumn> dbAllColumns = SqlExecutor.dbAllColumns(templateContext.tables());
        Map<String, List<TemplateColumn>> dbtables = dbAllColumns.stream()
                .collect(Collectors.groupingBy(e -> e.getTableName()));

        Map<String, TableDefinition> tables = new HashMap<>(dbtables.size());
        dbtables.forEach((table, columns) -> {
            TableDefinition tableDefinition = new TableDefinition(table, templateContext.prefixs());
            List<TableDefinition.ColumnDefinition> columnDefinitions = new ArrayList<>(columns.size());
            columns.forEach(e -> columnDefinitions.add(
                    new TableDefinition.ColumnDefinition(e.getColumnName(), e.getColumnType(), e.getColumnComment(), e.isNullable(), e.isPrimaryKey()))
            );
            tableDefinition.setColumns(columnDefinitions);
            tables.put(table, tableDefinition);
        });
        return tables;
    }

    @Override
    protected void tableProcess(String table, TableDefinition context) throws MockException {
        List<MethodDefinition> methodDefinitions = methodDefinitions(context);

        for (GenerateType generateType : GenerateType.values()) {
            if (generateType.enabled()) {
                print("│   ├── table=%s 使用模板%s生成代码文件.", table, generateType.freemarker());
                generateFile(generateType, context, methodDefinitions);
            }
        }
    }

    /**
     * 生成模板文件
     *
     * @param generateType
     *          生成文件类型 {@link GenerateType}
     * @param tableDefinition
     *          表定义实体
     * @param methodDefinitions
     *          方法定义
     */
    private void generateFile(GenerateType generateType, TableDefinition tableDefinition, List<MethodDefinition> methodDefinitions) {
        String pathname = generateType.type() == FileType.XML ? resourcesPath + generateType.pkgName() : javaPath + generateType.pkgName();
        File directoryFile = new File(pathname);
        if (!directoryFile.exists()) {
            directoryFile.mkdirs();
        }

        Map<String, Object> dataModel = new HashMap<>();
        dataModel.put("tableDefinition", tableDefinition);
        dataModel.put("methodDefinitions", methodDefinitions);
        dataModel.put("templateContext", templateContext);
        dataModel.putAll(packageMap);

        FreemarkerUtils.generate(
                pathname + File.separator + tableDefinition.getModelName() + generateTypeMap.get(generateType) + generateType.type().type(),
                generateType.freemarker(),
                dataModel
        );
    }

    /**
     * 获取方法定义
     *
     * @param tableDefinition
     *      表定义实体
     * @return
     *      方法定义集合
     */
    private List<MethodDefinition> methodDefinitions(TableDefinition tableDefinition) {
        List<MethodDefinition> methodDefinitions = new ArrayList<>();
        String camelModelName = tableDefinition.getCamelModelName();
        // 查询列表
        MethodDefinition queryList = new MethodDefinition(
                new MethodDefinition.ReturnType(List.class, tableDefinition.getModelName()), "queryList", "查询列表"
        );
        queryList.setMethodParams(
                Arrays.asList(
                        new MethodDefinition.Param(MockModel.class, camelModelName, "实例对象", tableDefinition.getModelName())
                )
        );
        methodDefinitions.add(queryList);

        // 根据主键查询
        MethodDefinition queryByPrimaryKey = new MethodDefinition(
                new MethodDefinition.ReturnType(MockModel.class, tableDefinition.getModelName()), "queryByPrimaryKey", "根据主键查询"
        );
        List<MethodDefinition.Param> primaryKeyParams = new ArrayList<>();
        for (TableDefinition.ColumnDefinition primaryKey : tableDefinition.getPrimaryKeys()) {
            primaryKeyParams.add(
                    new MethodDefinition.Param(
                            primaryKey.getDataType().classType(), primaryKey.getPropertyName(), primaryKey.getColumnComment(), tableDefinition.getModelName()
                    )
            );
        }
        queryByPrimaryKey.setMethodParams(primaryKeyParams);
        methodDefinitions.add(queryByPrimaryKey);

        // 新增一条记录
        MethodDefinition insert = new MethodDefinition(
                new MethodDefinition.ReturnType(Integer.class, tableDefinition.getModelName()), "insert", "新增一条记录"
        );
        insert.setMethodParams(
                Arrays.asList(
                        new MethodDefinition.Param(MockModel.class, camelModelName, "实例对象", tableDefinition.getModelName())
                )
        );
        methodDefinitions.add(insert);

        // 批量新增记录
        MethodDefinition batchInsert = new MethodDefinition(
                new MethodDefinition.ReturnType(Integer.class, tableDefinition.getModelName()), "batchInsert", "批量新增记录"
        );
        batchInsert.setMethodParams(
                Arrays.asList(
                        new MethodDefinition.Param(List.class, "list", "实例对象列表", tableDefinition.getModelName())
                )
        );
        methodDefinitions.add(batchInsert);

        // 更新记录
        MethodDefinition update = new MethodDefinition(
                new MethodDefinition.ReturnType(Integer.class, tableDefinition.getModelName()), "update", "更新记录"
        );
        update.setMethodParams(
                Arrays.asList(
                        new MethodDefinition.Param(MockModel.class, camelModelName, "实例对象", tableDefinition.getModelName())
                )
        );
        methodDefinitions.add(update);


        // 更新记录
        MethodDefinition deleteByPrimaryKey = new MethodDefinition(
                new MethodDefinition.ReturnType(Integer.class, tableDefinition.getModelName()), "deleteByPrimaryKey", "根据主键删除记录"
        );
        deleteByPrimaryKey.setMethodParams(primaryKeyParams);
        methodDefinitions.add(deleteByPrimaryKey);

        return methodDefinitions;
    }

    @Override
    public void close() throws Exception {
        if (generateTypeMap != null) {
            generateTypeMap.clear();
            generateTypeMap = null;
        }

        if (packageMap != null) {
            packageMap.clear();
            packageMap = null;
        }
    }
}