/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.core;

import java.util.ArrayList;
import java.util.List;

/**
 * Template
 *
 * @author Jaemon
 * @since 1.0
 */
public class TemplateContext {
    private String basePackage;
    /** 实体文件名 */
    private String entity;
    /** service文件名后缀， 默认： Service */
    private String service;
    /** serviceImpl文件名后缀， 默认： ServiceImpl */
    private String serviceImpl;
    /** dao文件名后缀， 默认： Mapper */
    private String dao;
    /** mapper文件名后缀， 默认： Mapper */
    private String mapper;
    /** 去掉表名前/后缀, sys_user去掉前缀sys_后为user */
    private List<String> prefixs;
    /** 指定表集合 */
    private List<String> tables;

    TemplateContext(
            final String basePackage, final String entity,
            final String service, final String serviceImpl,
            final String dao, final String mapper,
            final List<String> prefixs, final List<String> tables
    ) {
        this.basePackage = basePackage;
        this.entity = entity;
        this.service = service;
        this.serviceImpl = serviceImpl;
        this.dao = dao;
        this.mapper = mapper;
        this.prefixs = prefixs;
        this.tables = tables;
    }

    public String getBasePackage() {
        return basePackage;
    }

    public String getEntity() {
        return entity;
    }

    public String getService() {
        return service;
    }

    public String getServiceImpl() {
        return serviceImpl;
    }

    public String getDao() {
        return dao;
    }

    public String getMapper() {
        return mapper;
    }

    public List<String> prefixs() {
        return prefixs;
    }

    public List<String> tables() {
        return tables;
    }

    public static TemplateContext.TemplateBuilder builder() {
        return new TemplateContext.TemplateBuilder();
    }

    public static TemplateContext init(String basePackage) {
        return TemplateContext.builder().build(basePackage);
    }

    @Override
    public String toString() {
        return "Template(" +
                "basePackage=" + this.basePackage + ", entity=" + this.entity + ", " +
                "service=" + this.service + ", serviceImpl=" + this.serviceImpl + ", " +
                "dao=" + this.dao + ", mapper=" + this.mapper + ", " +
                "prefix=" + prefixs + ", tables=" + this.tables +
                ")";
    }

    public static class TemplateBuilder {
        private String entity = "";
        private String service = "Service";
        private String serviceImpl = "ServiceImpl";
        private String dao = "Mapper";
        private String mapper = "Mapper";
        private List<String> prefixs = new ArrayList<>();
        private List<String> tables = new ArrayList<>();

        TemplateBuilder() {
        }

        public TemplateContext.TemplateBuilder entity(final String entity) {
            this.entity = entity;
            return this;
        }

        public TemplateContext.TemplateBuilder service(final String service) {
            this.service = service;
            return this;
        }

        public TemplateContext.TemplateBuilder serviceImpl(final String serviceImpl) {
            this.serviceImpl = serviceImpl;
            return this;
        }

        public TemplateContext.TemplateBuilder dao(final String dao) {
            this.dao = dao;
            return this;
        }

        public TemplateContext.TemplateBuilder mapper(final String mapper) {
            this.mapper = mapper;
            return this;
        }

        public TemplateContext.TemplateBuilder prefixs(final List<String> prefixs) {
            this.prefixs = prefixs;
            return this;
        }

        public TemplateContext.TemplateBuilder tables(final List<String> tables) {
            this.tables = tables;
            return this;
        }

        public TemplateContext build(String basePackage) {
            return new TemplateContext(
                    basePackage, this.entity,
                    this.service, this.serviceImpl,
                    this.dao, this.mapper,
                    this.prefixs, this.tables
            );
        }

        @Override
        public String toString() {
            return "Template.TemplateBuilder(" +
                    "service=" + this.service + "" +
                    "serviceImpl=" + this.serviceImpl + "" +
                    "dao=" + this.dao + "" +
                    "mapper=" + this.mapper + "" +
                    "prefixs=" + this.prefixs + "" +
                    "tables=" + this.tables
                    + ")";
        }
    }


}