/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.core;

import io.gitee.jaemon.mocker.exception.MockException;
import io.gitee.jaemon.mocker.entity.Column;
import io.gitee.jaemon.mocker.entity.TemplateColumn;

import java.sql.*;
import java.util.ArrayList;
import java.util.List;

import static io.gitee.jaemon.mocker.core.SqlTemplate.*;

/**
 * 操作sql执行器
 *
 * @author Jaemon
 * @since 1.0
 */
public class SqlExecutor {
    protected static final SqlConfig sqlConfig = SqlConfig.sqlConfig();
    protected static Connection conn;
    static {
        try {
            Class.forName(sqlConfig.getDriver());
            conn = DriverManager.getConnection(sqlConfig.getUrl(), sqlConfig.getUname(), sqlConfig.getPassword());
        } catch (Exception e) {
            throw new MockException("获取连接失败~~~");
        }
    }

    public static final String db = sqlConfig.getDb();

    /**
     * 获取db的所有表名
     *
     * @param tableNames
     *      指定表集合
     * @return
     *      表名集合
     * */
    public static List<String> tableNamesInDb(List<String> tableNames) {
        List<String> tables = new ArrayList<>();
        try(
                Statement statement = conn.createStatement();
                ResultSet resultSet = statement.executeQuery(
                        String.format(DB_TABLES_SQL, db, where(tableNames))
                )
        ) {
            while (resultSet.next()) {
                tables.add(resultSet.getString(1));
            }

        } catch (Exception ex) {
            throw new MockException("数据库操作失败!!!");
        }
        return tables;
    }


    /**
     * 获取db的所有字段信息
     *
     * @param tableNames
     *      指定表集合
     * @return
     *      字段信息集合
     * */
    public static List<Column> dbColumns(List<String> tableNames) {
        List<Column> tables = new ArrayList<>();
        try(
                Statement statement = conn.createStatement();
                ResultSet resultSet = statement.executeQuery(
                        String.format(DB_COLUMNS_SQL, db, where(tableNames))
                )
        ) {
            while (resultSet.next()) {
                tables.add(
                        new Column(
                                resultSet.getString(1),
                                resultSet.getString(2),
                                resultSet.getString(3)
                        )
                );
            }

        } catch (Exception ex) {
            throw new MockException("数据库操作失败!!!");
        }
        return tables;
    }


    /**
     * 获取db的所有字段信息
     *
     * @param tableNames
     *      指定表集合
     * @return
     *      字段信息集合
     * */
    public static List<TemplateColumn> dbAllColumns(List<String> tableNames) {
        List<TemplateColumn> tables = new ArrayList<>();
        try(
                Statement statement = conn.createStatement();
                ResultSet resultSet = statement.executeQuery(
                        String.format(DB_ALL_COLUMNS_SQL, db, where(tableNames))
                )
        ) {
            while (resultSet.next()) {
                tables.add(
                        new TemplateColumn(
                                resultSet.getString(1),
                                resultSet.getString(2),
                                resultSet.getString(3),
                                resultSet.getString(4),
                                resultSet.getBoolean(5),
                                resultSet.getBoolean(6)
                        )
                );
            }

        } catch (Exception ex) {
            throw new MockException("数据库操作失败!!!");
        } finally {
            try {
                conn.close();
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
        return tables;
    }

    /**
     * 获取表的所有字段
     *
     * @param tableName
     *          表名
     * @return
     *          表字段集合
     */
    public static List<Column> tableColumns(String tableName) {
        String sql = String.format(TABLE_COLUMNS_SQL, db, tableName);
        List<Column> columns = new ArrayList<>();

        try(
                Statement statement = conn.createStatement();
                ResultSet resultSet = statement.executeQuery(sql)
        ) {

            while (resultSet.next()) {
                columns.add(new Column(
                        resultSet.getString(1),
                        resultSet.getInt(2)
                ));
            }
        } catch (Exception ex) {
            throw new MockException("数据库操作失败!!!");
        }
        return columns;
    }

    /**
     * 执行sql插入
     *
     * @param sql
     *      插入的sql语句
     * @return
     *      影响结果数
     */
    public static int insert(String sql) {
        try(
                Statement statement = conn.createStatement();
        ) {
            return statement.executeUpdate(sql);
        } catch (Exception e) {
            System.out.println(String.format("sql=%s, msg=%s.", sql, e.getMessage()));
            return 0;
        }
    }

    /**
     * 批量执行sql插入
     *
     * @param sqls
     *      批量sql语句集
     * @return
     *      影响结果数
     */
    public static int[] insertBatch(String... sqls) {
        try {
            conn.setAutoCommit(false);
        } catch (SQLException e) {
            throw new MockException("");
        }
        try(
                Statement statement = conn.createStatement();
        ) {
            for (String sql : sqls) {
                statement.addBatch(sql);
            }
            int[] count = statement.executeBatch();
            return count;
        } catch (Exception e) {
            System.out.println(String.format("msg=%s.", e.getMessage()));
            return new int[0];
        } finally {
            try {
                conn.setAutoCommit(true);
            } catch (SQLException e) {
                throw new MockException("");
            }
        }
    }


    /**
     * 执行sql插入
     *
     * @param sql
     *      插入的sql语句-insert into common_user (name) values(?)
     * @return
     *      影响结果数
     */
    public static long insertKey(String sql) {
        try(
                PreparedStatement preparedStatement = conn.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
        ) {
            preparedStatement.executeUpdate(sql);
//            preparedStatement.setString(1, null);
            ResultSet generatedKeys = preparedStatement.getGeneratedKeys();
            if (generatedKeys.next()) {
                return generatedKeys.getLong(1);
            }
        } catch (Exception e) {
            System.out.println(String.format("sql=%s, msg=%s.", sql, e.getMessage()));
        }
        return 0;
    }

    private static String where(List<String> tableNames) {
        StringBuilder where = new StringBuilder();

        if (tableNames.isEmpty()) {
            if (!sqlConfig.getIncludeTables().isEmpty()) {
                String include = String.join("','", sqlConfig.getIncludeTables());
                where.append(String.format(" and table_name in ('%s') ", include));
            }

            if (!sqlConfig.getExcludeTables().isEmpty()) {
                String exclude = String.join("", sqlConfig.getExcludeTables());
                where.append(String.format(" and table_name not in ('%s') ", exclude));
            }
        } else {
            String include = String.join("','", tableNames);
            where.append(String.format(" and table_name in ('%s') ", include));
        }

        return where.toString();
    }

}