/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.core;

import io.gitee.jaemon.mocker.TableColumnsHandler;
import io.gitee.jaemon.mocker.entity.MockData;
import io.gitee.jaemon.mocker.function.AutoIncrement;
import io.gitee.jaemon.mocker.function.Function;
import io.gitee.jaemon.mocker.function.InnerFunction;
import io.gitee.jaemon.mocker.exception.MockException;

import java.util.*;

import static io.gitee.jaemon.mocker.core.SqlExecutor.*;
import static io.gitee.jaemon.mocker.core.SqlExecutor.insertBatch;
import static io.gitee.jaemon.mocker.core.SqlTemplate.TRUNCATE_TABLE_SQL;

/**
 * sql数据处理器-写入到数据库表
 *
 * @author Jaemon
 * @since 1.0
 */
public class SqlDataProcessor extends AbstractDataMockExecutor {
    private static final String INSERT_PREFIX = "insert into `%s`(`%s`) values";

    public SqlDataProcessor(Class<? extends TableColumnsHandler> clazz, List<String> tables) {
        super(clazz, tables);
    }


    @Override
    protected void process(MockData mockData) throws MockException {
        List<String> columnNames = mockData.getColumns();
        String columns = String.join("`,`", columnNames);

        StringBuilder insertSql = new StringBuilder();
        insertSql.append(String.format(INSERT_PREFIX, mockData.getTableName(), columns));

        StringBuilder valueSql = new StringBuilder();
        int batchStartIndex = mockData.getBatchStartIndex();
        for (Map<String, Object> data : mockData.getDatas()) {
            valueSql.append("(");

            for (String columnName : columnNames) {
                Object value = data.get(columnName);
                if (String.class.isInstance(value)) {
                    valueSql.append("'" + value + "'").append(",");
                } else if (InnerFunction.class.isInstance(value)) {
                    valueSql.append(((InnerFunction)value).function()).append(",");
                } else if (AutoIncrement.class.isInstance(value)) {
                    int startValue = ((AutoIncrement) value).value();
                    value = startValue + batchStartIndex;
                    valueSql.append(value).append(",");
                } else if (Function.class.isInstance(value)) {
                    Object execute = ((Function) value).execute(batchStartIndex, mockData);
                    if (Number.class.isInstance(execute)) {
                        valueSql.append(execute).append(",");
                    } else {
                        valueSql.append("'" + execute + "'").append(",");
                    }
                } else {
                    valueSql.append(value).append(",");
                }
            }
            valueSql.deleteCharAt(valueSql.length() - 1);
            valueSql.append("),");
            batchStartIndex++;
        }

        valueSql.deleteCharAt(valueSql.length() - 1);

        insertSql.append(valueSql);
        if (sqlConfig.isTruncateTable()) {
            insertBatch(String.format(TRUNCATE_TABLE_SQL, mockData.getTableName()), insertSql.toString());
        } else {
            insertBatch(insertSql.toString());
        }
    }

}