/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.core;

import io.gitee.jaemon.mocker.entity.Column;
import io.gitee.jaemon.mocker.entity.eunms.FileType;
import io.gitee.jaemon.mocker.exception.MockException;
import io.gitee.jaemon.mocker.template.DataBaseDataType;
import io.gitee.jaemon.mocker.utils.FileUtils;

import java.io.File;
import java.io.IOException;
import java.util.List;

/**
 * 字段处理类生成
 *
 * @author Jaemon
 * @since 1.0
 */
public class SplicingGenerator extends ColumnHandlerGenerator {
    private static final String HEADER = "" +
            "package io.gitee.jaemon.mocker;\n\n" +
            "import io.gitee.jaemon.mocker.TableColumnsHandler;\n" +
            "import java.util.Map;\n" +
            "import java.util.Date;\n\n" +
            "public enum " + CLASS_NAME + " implements TableColumnsHandler { \n";

    private static final String COLUMN_HANDLER_BODY = "\t" +
            "// %s \n \t" +
            "%s { \n \t\t" +
            "@Override \n \t\t" +
            "public %s generate(String tableName, Map<String, Object> values) { \n \t\t\t" +
            "return null; \n\t\t" +
            "}\n\t" +
            "},\n";

    private static final String FOOTER = "\t;\n\n\t" + CLASS_NAME + "() {\n\n\t}\n\n}";
    private static final String COLUMN_HANDLER_TEXT = HEADER  + "%s" + FOOTER;

    public SplicingGenerator(List<String> tables) {
        super(tables);
    }


    @Override
    public String generate(String classPath) {
        StringBuilder columnsBody =  new StringBuilder();
        for (Column column : dbColumns) {
            columnsBody.append(
                    String.format(
                            COLUMN_HANDLER_BODY,
                            column.getColumnComment(),
                            column.getColumnName().toUpperCase(),
                            DataBaseDataType.matching(column.getDataType()).packType()
                    )
            );
        }

        if (!dbColumns.isEmpty()) {
            columnsBody.deleteCharAt(columnsBody.lastIndexOf(","));
        }
        columnsBody.append("\n\n");


        String columnHandlerText = String.format(COLUMN_HANDLER_TEXT, columnsBody.toString());
        try {
            String pathName = classPath + File.separator + CLASS_NAME + FileType.JAVA.type();
            FileUtils.writeStringToFile(
                    new File(pathName), columnHandlerText
            );
            return pathName;
        } catch (IOException e) {
            throw new MockException(e.getMessage());
        }
    }

}