/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker.core;

import io.gitee.jaemon.mocker.TableColumnsHandler;
import io.gitee.jaemon.mocker.entity.MockData;
import io.gitee.jaemon.mocker.common.Constants;
import io.gitee.jaemon.mocker.exception.MockException;
import io.gitee.jaemon.mocker.annotation.ColumnOrder;
import io.gitee.jaemon.mocker.annotation.TableColumn;
import io.gitee.jaemon.mocker.entity.Column;

import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static io.gitee.jaemon.mocker.core.SqlExecutor.*;
import static io.gitee.jaemon.mocker.utils.StringUtils.log;
import static io.gitee.jaemon.mocker.utils.StringUtils.print;

/**
 * 抽象mock执行器
 *
 * @author Jaemon
 * @since 1.0
 */
public abstract class AbstractDataMockExecutor implements MockGenerator {
    protected final List<String> dbTables;
    protected final Map<String, Map<String, Integer>> columnsOrderMap;

    public AbstractDataMockExecutor(Class<? extends TableColumnsHandler> clazz, List<String> tables) {
        ColumnHandlerFactory.register(clazz);

        if ((dbTables = tableNamesInDb(tables)).isEmpty()) {
            throw new MockException(log("数据库=%s不存在任何表.", sqlConfig.getDb()));
        }

        // 处理字段处理类配置
        columnsOrderMap = new HashMap<>();
        List<String> enumNames = Stream.of(clazz.getEnumConstants()).map(e -> e.name()).collect(Collectors.toList());
        List<Field> enumItems = Stream.of(clazz.getFields()).filter(
                e -> enumNames.contains(e.getName()) && e.isAnnotationPresent(ColumnOrder.class)
        ).collect(Collectors.toList());
        for (Field enumItem : enumItems) {
            String column_name = Constants.LOWER_UNDERSCORE_CONVERT.convert(enumItem.getName());
            ColumnOrder columnOrder = enumItem.getAnnotation(ColumnOrder.class);
            int value =  Math.max(columnOrder.value(), TableColumnsHandler.ORDINAL_POSITION);
            put(TableColumnsHandler.DEFAULT_TABLE_NAME, column_name, value);

            TableColumn[] tableColumns = columnOrder.order();
            // priority
            if (tableColumns.length > 0) {
                for (TableColumn tableColumn : tableColumns) {
                    String table = "".equals(tableColumn.table()) ? TableColumnsHandler.DEFAULT_TABLE_NAME : tableColumn.table();
                    int order = tableColumn.order();
                    put(table, column_name, order);
                }
            }
        }
    }

    @Override
    public void mock() throws MockException {
        List<String> validTables = tableNames();
        print("本次任务待模拟数据的表信息%s.", validTables);
        for (String table : validTables) {
            print("--table=%s 准备开始数据%s模拟~~", table, sqlConfig.isInBatches() ? "分批" : "全量");
            tableProcess(table);
        }
        print("本次任务完成表%s的数据模拟.", validTables);
    }


    /**
     * 表处理
     *
     * @param table
     *      表名
     * @throws MockException
     *      ex
     */
    private void tableProcess(String table) throws MockException {
        List<Column> tableColumns = tableColumns(table);
        columnsOrder(table, tableColumns);
        tableColumns.sort(Comparator.comparingInt(Column::getOrder));
        List<String> columnNames = tableColumns.stream().map(Column::getColumnName).collect(Collectors.toList());

        boolean batch = sqlConfig.isInBatches();
        int record = sqlConfig.getRecordCount();
        int per_record = 50;
        int count = (int) Math.ceil((double)record / per_record);
        int size = columnNames.size();
        if (batch) {
            print("--table=%s 模拟记录总数=%s, 批量条数=%s, 批次数=%s.", table, record, per_record, count);
        } else {
            print("--table=%s 模拟记录总数=%s.", table, record);
        }

        List<Map<String, Object>> pairs = batch ? null : new ArrayList<>();

        // 批量次数
        for (int i = 0; i < count; i++) {
            int start = i * per_record, end = start + per_record;
            end = Math.min(record, end);
            Map<String, Object> values;
            pairs = batch ? new ArrayList<>() : pairs;
            // 一次性条数
            for (int m = start; m < end; m++) {
                values = new HashMap<>(size);
                // 一条记录
                for (int index = 0; index < size; index++) {
                    String column = columnNames.get(index);
                    TableColumnsHandler handler = ColumnHandlerFactory.get(column);
                    Object value = null;
                    if (handler != null) {
                        value = handler.generate(table, values);
                    }

                    values.put(column, value);
                }
                pairs.add(values);
            }
            if (batch && !pairs.isEmpty()) {
                print("----table=%s 进行第%s次共%s条记录批数据处理.", table, (i + 1), pairs.size());
                process(MockData.mockData(table, columnNames, start, pairs));
            }
        }

        if (!batch && !pairs.isEmpty()) {
            print("--table=%s 预计模拟%s条，实际模拟%s条记录批数据处理.", table, record, pairs.size());
            process(MockData.mockData(table, columnNames, 0, pairs));
        }

    }

    /**
     * 数据处理
     *
     * @param mockData
     *          数据信息 {@link MockData}
     * @throws MockException
     *          ex
     */
    protected abstract void process(MockData mockData) throws MockException;


    private List<String> tableNames() throws MockException {
        // 校验字段处理类中表信息
        List<String> invalidConfigTables = columnsOrderMap.keySet().stream().filter(
                e -> e != TableColumnsHandler.DEFAULT_TABLE_NAME && !dbTables.contains(e)
        ).collect(Collectors.toList());
        if (!invalidConfigTables.isEmpty()) {
            throw new MockException(log("表字段处理类配置了无效表名 %s.", invalidConfigTables));
        }

        if (dbTables.isEmpty()) {
            throw new MockException("当前任务无有效表信息.");
        }

        return dbTables;
    }

    /**
     * 字段解析序号设置
     *
     * @param tableName
     *      表名
     * @param columns
     *      表字段集合
     */
    protected void columnsOrder(String tableName, List<Column> columns) {
        Map<String, Integer> tableMap = columnsOrderMap.getOrDefault(tableName, new HashMap<>());
        Map<String, Integer> defaultMap = columnsOrderMap.getOrDefault(TableColumnsHandler.DEFAULT_TABLE_NAME, new HashMap<>());

        for (Column column : columns) {
            String columnName = column.getColumnName();
            if (defaultMap.containsKey(columnName)) {
                column.setOrder(defaultMap.get(columnName));
            }

            if (tableMap.containsKey(columnName)) {
                column.setOrder(tableMap.get(columnName));
            }
        }
    }

    private void put(String key, String hkey, int value) {
        if (value > TableColumnsHandler.ORDINAL_POSITION) {
            Map<String, Integer> valueMap = columnsOrderMap.getOrDefault(key, new HashMap<>());
            valueMap.put(hkey, value);
            if (!columnsOrderMap.containsKey(key)) {
                columnsOrderMap.put(key, valueMap);
            }
        }
    }

    @Override
    public void close() throws Exception {
        if (conn != null) {
            conn.close();
        }
        if (columnsOrderMap != null) {
            columnsOrderMap.clear();
        }
        if (dbTables != null) {
            dbTables.clear();
        }
    }
}