/*
 * Copyright ©2015-2021 Jaemon. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.gitee.jaemon.mocker;

import io.gitee.jaemon.mocker.core.*;
import io.gitee.jaemon.mocker.utils.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;


/**
 * 数据mock启动类
 *
 * @author Jaemon
 * @since 1.0
 */
public class DataMockStarter {

    /**
     * mock表数据
     *
     * <pre>
     *     需先执行{@link DataMockStarter#generate}生成字段处理类{@link TableColumnsHandler#generate(String, Map)}
     * </pre>
     *
     * @param clazz
     *          表字段处理类
     */
    public static void mock(Class<? extends TableColumnsHandler> clazz) {
        mock(clazz, new ArrayList<>());
    }

    /**
     * mock表数据
     *
     * <pre>
     *     需先执行{@link DataMockStarter#generate}生成字段处理类{@link TableColumnsHandler#generate(String, Map)}
     * </pre>
     *
     * @param clazz
     *          表字段处理类
     * @param tables
     *          指定表
     */
    public static void mock(Class<? extends TableColumnsHandler> clazz, List<String> tables) {
        try (
                MockGenerator mockGenerator = new SqlDataProcessor(clazz, tables)
        ) {
            mockGenerator.mock();
        } catch (Exception ex) {
            StringUtils.print("%s", ex.getMessage());
        }
    }


    /**
     * mock表数据
     *
     * <pre>
     *     需先执行{@link DataMockStarter#generate}生成字段处理类{@link TableColumnsHandler#generate(String, Map)}
     * </pre>
     *
     * @param clazz
     *          表字段处理类
     * @param dataProcessor
     *          数据处理器
     */
    public static void mock(
            Class<? extends TableColumnsHandler> clazz,
            Class<? extends AbstractDataMockExecutor> dataProcessor
    ) {
        mock(clazz, new ArrayList<>(), dataProcessor);
    }

    /**
     * mock表数据
     *
     * <pre>
     *     需先执行{@link DataMockStarter#generate}生成字段处理类{@link TableColumnsHandler#generate(String, Map)}
     * </pre>
     *
     * @param clazz
     *          表字段处理类
     * @param tables
     *          指定表
     * @param dataProcessor
     *          数据处理器
     */
    public static void mock(
            Class<? extends TableColumnsHandler> clazz,
            List<String> tables,
            Class<? extends AbstractDataMockExecutor> dataProcessor
    ) {
        try (
                MockGenerator mockGenerator =
                        dataProcessor
                                .getConstructor(Class.class, List.class)
                                .newInstance(clazz, tables)
        ) {
            mockGenerator.mock();
        } catch (Exception ex) {
            StringUtils.print("%s", ex.getMessage());
        }
    }

    /**
     * 生成字段处理类
     *
     * @param filePath
     *      类生成路径, /home/jaemon
     *
     * @see TableColumnsHandler#generate(String, Map)
     */
    public static void generate(String filePath) {
        generate(filePath, new ArrayList<>());
    }

    /**
     * 生成字段处理类
     *
     * @param filePath
     *      类生成路径, /home/jaemon
     * @param tables
     *      指定表集合
     *
     * @see TableColumnsHandler#generate(String, Map)
     */
    public static void generate(String filePath, List<String> tables) {
        try (
                ColumnHandlerGenerator generator = new TemplateGenerator(tables)
        ) {
            String path = generator.generate(filePath);
            StringUtils.print("字段处理类已生成.%s.", path);
        } catch (Exception ex) {
            StringUtils.print("%s", ex.getMessage());
        }
    }

    /**
     * 生成字段处理类
     *
     * @param filePath
     *      类生成路径, /home/jaemon
     * @param tables
     *      指定表集合
     * @param clazz
     *      生成器类
     *
     * @see TableColumnsHandler#generate(String, Map)
     */
    public static void generate(
            String filePath,
            List<String> tables,
            Class<? extends ColumnHandlerGenerator> clazz
    ) {
        try (
                ColumnHandlerGenerator generator = clazz.getConstructor(List.class).newInstance(tables)
        ) {
            String path = generator.generate(filePath);
            StringUtils.print("字段处理类已生成.%s.", path);
        } catch (Exception ex) {
            StringUtils.print("%s", ex.getMessage());
        }
    }
}