package com.jaemon.commons.toolkit.zip;

import com.jaemon.commons.toolkit.StringUtils;
import lombok.extern.slf4j.Slf4j;
import net.lingala.zip4j.ZipFile;
import net.lingala.zip4j.exception.ZipException;
import net.lingala.zip4j.model.FileHeader;
import net.lingala.zip4j.model.ZipParameters;
import net.lingala.zip4j.model.enums.CompressionLevel;
import net.lingala.zip4j.model.enums.CompressionMethod;
import net.lingala.zip4j.model.enums.EncryptionMethod;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

/**
 * zip 包压缩解压工具类
 *
 * @author Jaemon
 */
@Slf4j
public class ZipUtils {

    private ZipUtils() {}

    /**
     * 压缩
     *
     * @param zipFilePath 压缩包文件路径
     * @param zipFiles 待压缩文件列表
     * @throws ZipException ze
     */
    public static void zip(String zipFilePath, List<File> zipFiles) throws ZipException {
        zip(zipFilePath, zipFiles, null);
    }

    /**
     * 压缩
     *
     * @param zipFilePath 压缩包文件路径
     * @param zipFiles 待压缩文件列表
     * @param pwd 设置压缩包密码(注意待压缩文件列表中包含文件时才有效, 如果只是包含目录则密码设置无效)
     * @throws ZipException ze
     */
    public static void zip(String zipFilePath, List<File> zipFiles, String pwd) throws ZipException {
        zipFiles.removeIf(file -> {
            boolean notExists = !file.exists();
            if (notExists) {
                log.warn("file=[{}] is not exists", file.getPath());
            }
            return notExists;
        });

        if (zipFiles.size() < 1) {
            log.warn("file size is zero");
            return;
        }

        // 如果压缩包目录不存在则新建
        String zipParentPath = new File(zipFilePath).getParent();
        if (!new File(zipParentPath).exists()) {
            new File(zipParentPath).mkdirs();
        }

        ZipFile zipFile;
        ZipParameters parameters = new ZipParameters();
        parameters.setCompressionMethod(CompressionMethod.DEFLATE);
        parameters.setCompressionLevel(CompressionLevel.NORMAL);

        if (StringUtils.isEmpty(pwd)) {
            zipFile = new ZipFile(zipFilePath);
        } else {
            zipFile = new ZipFile(zipFilePath, pwd.toCharArray());
            // set password
            parameters.setEncryptFiles(true);
            parameters.setEncryptionMethod(EncryptionMethod.ZIP_STANDARD);
//            parameters.setEncryptionMethod(EncryptionMethod.AES);
//            parameters.setAesKeyStrength(AesKeyStrength.KEY_STRENGTH_256);
        }

        zipFile.addFiles(zipFiles, parameters);

        log.info("zipFile is Encrypted={}.", zipFile.isEncrypted());
    }



    /**
     * 分卷压缩
     *
     * @param zipFilePath 压缩包文件路径
     * @param zipFiles 待压缩文件列表
     * @throws ZipException ze
     */
    public static void zipSplit(String zipFilePath, List<File> zipFiles) throws ZipException {
        zipSplit(zipFilePath, zipFiles, 65535, null);
    }


    /**
     * 分卷压缩
     *
     * @param zipFilePath 压缩包文件路径
     * @param zipFiles 待压缩文件列表
     * @param pwd 设置压缩密码
     * @throws ZipException ze
     */
    public static void zipSplit(String zipFilePath, List<File> zipFiles, String pwd) throws ZipException {
        zipSplit(zipFilePath, zipFiles, 65535, pwd);
    }

    /**
     * 分卷压缩
     *
     * @param zipFilePath 压缩包文件路径
     * @param zipFiles 待压缩文件列表
     * @param splitLength 分卷大小(<b>splitLength minimum: 65536</b>)
     * @throws ZipException ze
     * */
    public static void zipSplit(String zipFilePath, List<File> zipFiles, long splitLength) throws ZipException {
        zipSplit(zipFilePath, zipFiles, splitLength, null);
    }

    /**
     * 分卷压缩
     *
     * @param zipFilePath 压缩包文件路径
     * @param zipFiles 待压缩文件列表
     * @param splitLength 分卷大小(<b>splitLength minimum: 65536</b>)
     * @param pwd 设置压缩密码
     * @throws ZipException ze
     * */
    public static void zipSplit(String zipFilePath, List<File> zipFiles, long splitLength, String pwd) throws ZipException {
        ZipFile zipFile;

        ZipParameters parameters = new ZipParameters();
        parameters.setCompressionMethod(CompressionMethod.DEFLATE);
        parameters.setCompressionLevel(CompressionLevel.NORMAL);

        // 如果压缩包目录不存在则新建
        String zipParentPath = new File(zipFilePath).getParent();
        if (!new File(zipParentPath).exists()) {
            new File(zipParentPath).mkdirs();
        }

        if (StringUtils.isEmpty(pwd)) {
            zipFile = new ZipFile(zipFilePath);
        } else {
            zipFile = new ZipFile(zipFilePath, pwd.toCharArray());

            // set password
            parameters.setEncryptFiles(true);
            parameters.setEncryptionMethod(EncryptionMethod.ZIP_STANDARD);
        }

        // splitLength minimum: 65536 bytes = 1024 bytes * 64 = 64K
        splitLength = splitLength < 65536 ? 65536 : splitLength;
        zipFile.createSplitZipFile(zipFiles, parameters, true, splitLength);
    }



    /**
     * 压缩(将文件以流的形式添加到压缩包)
     *
     * @param zipFilePath 压缩包文件路径
     * @param filePath 待压缩的文件路径
     * @throws IOException ie
     */
    public static void zipStream(String zipFilePath, String filePath) throws IOException {
        File file = new File(filePath);
        zipStream(zipFilePath, file, file.getName(), null);
    }

    /**
     * 压缩(将文件以流的形式添加到压缩包)
     *
     * @param zipFilePath 压缩包文件路径
     * @param filePath 待压缩的文件路径
     * @param pwd 设置压缩密码
     * @throws IOException ie
     */
    public static void zipStreamPwd(String zipFilePath, String filePath, String pwd) throws IOException {
        File file = new File(filePath);
        zipStream(zipFilePath, file, file.getName(), pwd);
    }

    /**
     * 压缩(将文件以流的形式添加到压缩包)
     *
     * @param zipFilePath 压缩包文件路径
     * @param filePath 待压缩的文件路径
     * @param fileNameInZip 文件在压缩包内的文件名
     * @throws IOException ie
     */
    public static void zipStream(String zipFilePath, String filePath, String fileNameInZip) throws IOException {
        zipStream(zipFilePath, new File(filePath), fileNameInZip, null);
    }

    /**
     * 压缩(将文件以流的形式添加到压缩包)
     *
     * @param zipFilePath 压缩包文件路径
     * @param filePath 待压缩的文件路径
     * @param fileNameInZip 文件在压缩包内的文件名
     * @param pwd 设置压缩密码
     * @throws IOException ie
     */
    public static void zipStream(String zipFilePath, String filePath, String fileNameInZip, String pwd) throws IOException {
        zipStream(zipFilePath, new File(filePath), fileNameInZip, pwd);
    }

    /**
     * 压缩(将文件以流的形式添加到压缩包)
     *
     * @param zipFilePath 压缩包文件路径
     * @param file 待压缩的文件
     * @param fileNameInZip 文件在压缩包内的文件名
     * @throws IOException ie
     */
    public static void zipStream(String zipFilePath, File file, String fileNameInZip) throws IOException {
        zipStream(zipFilePath, file, fileNameInZip, null);
    }

    /**
     * 压缩(将文件以流的形式添加到压缩包)
     *
     * @param zipFilePath 压缩包文件路径
     * @param file 待压缩的文件
     * @param fileNameInZip 文件在压缩包内的文件名
     * @param pwd 设置压缩密码
     * @throws IOException ie
     */
    public static void zipStream(String zipFilePath, File file, String fileNameInZip, String pwd) throws IOException {
        if (!file.exists()) {
//            log.info("file[{}] is not exists", file.getPath());
            return;
        }

        try (InputStream is = new FileInputStream(file)) {
            ZipFile zipFile;
            ZipParameters parameters = new ZipParameters();
            // 使用的压缩算法, DEFLATE 默认压缩算法, 如果不想使用任何压缩算法, 可用 CompressionMethod.STORE
            parameters.setCompressionMethod(CompressionMethod.DEFLATE);

            // 如果压缩包目录不存在则新建
            String zipParentPath = new File(zipFilePath).getParent();
            if (!new File(zipParentPath).exists()) {
                new File(zipParentPath).mkdirs();
            }

            if (StringUtils.isEmpty(pwd)) {
                zipFile = new ZipFile(zipFilePath);
            } else {
                zipFile = new ZipFile(zipFilePath, pwd.toCharArray());

                // set password
                parameters.setEncryptFiles(true);
                parameters.setEncryptionMethod(EncryptionMethod.ZIP_STANDARD);
            }

            fileNameInZip = StringUtils.isNotEmpty(fileNameInZip) ? fileNameInZip : file.getName();
            parameters.setFileNameInZip(fileNameInZip);

            zipFile.addStream(is, parameters);
        }
    }



    /**
     * 解压(有些情况下不支持文件名称为中文)
     *
     * @param srcPath 待解压的 zip 文件路径
     * @param descPath 解压路径
     * @throws ZipException ze
     */
    public static void unzipEn(String srcPath, String descPath) throws ZipException {
        unzipEn(new File(srcPath), descPath, null);
    }

    /**
     * 解压(有些情况下不支持文件名称为中文)
     *
     * @param srcPath 待解压的 zip 文件路径
     * @param descPath 解压路径
     * @param pwd 设置解压密码
     * @throws ZipException ze
     */
    public static void unzipEn(String srcPath, String descPath, String pwd) throws ZipException {
        unzipEn(new File(srcPath), descPath, pwd);
    }

    /**
     * 解压(有些情况下不支持文件名称为中文)
     *
     * @param srcFile 待解压的 zip 文件
     * @param descPath 解压路径
     * @throws ZipException ze
     */
    public static void unzipEn(File srcFile, String descPath) throws ZipException {
        unzipEn(srcFile, descPath, null);
    }

    /**
     * 解压(有些情况下不支持文件名称为中文)
     *
     * @param srcFile 待解压的 zip 文件
     * @param descPath 解压路径
     * @param pwd 设置解压密码
     * @throws ZipException ze
     */
    public static void unzipEn(File srcFile, String descPath, String pwd) throws ZipException {
        String srcPath = srcFile.getPath();
        if (!srcFile.exists()) {
            log.info("file=[{}] is not exists", srcPath);
            return;
        }

        ZipFile zipFile = new ZipFile(srcFile);

        if (!zipFile.isValidZipFile()) {
            log.info("file=[{}] is not valid zip file", srcPath);
            return;
        }

        if (zipFile.isEncrypted()) {
            zipFile = new ZipFile(srcFile, pwd.toCharArray());
        }

        zipFile.extractAll(descPath);
    }



    /**
     * 解压
     *
     * @param srcPath 待解压的 zip 文件路径
     * @param descPath 解压路径
     * @throws ZipException ze
     */
    public static void unzip(String srcPath, String descPath) throws ZipException {
        File file = new File(srcPath);
        unzip(file, descPath, null);
    }

    /**
     * 解压
     *
     * @param srcFile 待解压的 zip 文件路径
     * @param descPath 解压路径
     * @throws ZipException ze
     */
    public static void unzip(File srcFile, String descPath) throws ZipException {
        unzip(srcFile, descPath, null);
    }

    /**
     * 解压
     *
     * @param srcPath 待解压的 zip 文件路径路径
     * @param descPath 解压路径
     * @param password zip 包是否设置密码
     * @throws ZipException ze
     */
    public static void unzip(String srcPath, String descPath, String password) throws ZipException {
        File file = new File(srcPath);
        unzip(file, descPath, password);
    }


    /**
     * 解压
     *
     * @param srcFile 待解压的 zip 文件路径
     * @param descPath 解压路径
     * @param password zip 包是否设置密码
     * @throws ZipException ze
     */
    public static void unzip(File srcFile, String descPath, String password) throws ZipException {
        String srcPath = srcFile.getPath();
        if (!srcFile.exists()) {
            log.info("file=[{}] is not exists", srcPath);
            return;
        }

        ZipFile zipFile = new ZipFile(srcPath);

        if (!zipFile.isValidZipFile()) {
            log.info("file=[{}] is not valid zip file", srcPath);
            return;
        }

        if (zipFile.isEncrypted()) {
            if (StringUtils.isEmpty(password)) {
                System.out.println("该压缩包设置了密码, 请输入密码");
                return;
            }
            zipFile = new ZipFile(srcFile, password.toCharArray());
        }

        List<FileHeader> fileHeaders = zipFile.getFileHeaders();
        for (FileHeader fileHeader : fileHeaders) {
            String fileName = fileHeader.getFileName();
            // https://github.com/srikanth-lingala/zip4j#faq // Are unicode file names supported?
            // 解决中文文件名乱码问题
//            String fileName = new String(fileHeader.getFileName().getBytes("CP437"), "gbk");
                log.info("fileName=[{}]", fileName);
            zipFile.extractFile(fileHeader, descPath, fileName);
        }
    }


    private static ZipFile zipFile(File file, String pwd) {
        ZipFile zipFile;
        if (StringUtils.isEmpty(pwd)) {
            zipFile = new ZipFile(file);
        } else {
            zipFile = new ZipFile(file, pwd.toCharArray());
        }
        return zipFile;
    }

}