package com.jaemon.commons.toolkit;

import com.jaemon.commons.toolkit.enums.ImageFormatEnum;
import com.jaemon.commons.toolkit.exception.PdfOptException;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.rendering.ImageType;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.icepdf.core.exceptions.PDFException;
import org.icepdf.core.exceptions.PDFSecurityException;
import org.icepdf.core.pobjects.Document;
import org.icepdf.core.pobjects.Page;
import org.icepdf.core.util.GraphicsRenderingHints;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;


/**
 * PDF 文件转图片工具类
 *
 * @author Jaemon
 */
public class PdfUtils {

    private PdfUtils() {}


    /**
     * PDF 文件转多张图片(按页转)
     *
     * @param filePath pdf 文件路径
     * @return list
     */
    public static List<String> pdf2Images(String filePath){
        // 获取去除后缀的文件路径
        String descPath = filePath.substring(0, filePath.lastIndexOf("."));

        return pdf2Images(filePath, descPath);
    }

    /**
     * PDF 文件转多张图片(按页转)
     *
     * @param filePath pdf 文件路径
     * @param descPath 转换后图片存储路径
     * @return list
     */
    public static List<String> pdf2Images(String filePath, String descPath){
        List<String> list = new ArrayList<>();

        String imagePath;
        File file = new File(filePath);
        try {
            File descFile = new File(descPath);
            if(!descFile.exists()){
                descFile.mkdirs();
            }
            PDDocument pdf = PDDocument.load(file);
            PDFRenderer renderer = new PDFRenderer(pdf);
            int pageCount = pdf.getNumberOfPages();
            for (int i = 0; i < pageCount; i++) {
                // 方式1, 第二个参数是设置缩放比(即像素)
                // BufferedImage image = renderer.renderImageWithDPI(i, 296);
                // 方式2, 第二个参数是设置缩放比(即像素) 第二个参数越大生成图片分辨率越高，转换时间也就越长
                BufferedImage image = renderer.renderImage(i, 1.5f);
                imagePath = descPath + File.separator + i + ImageFormatEnum.JPG.suffix();
                ImageIO.write(image, ImageFormatEnum.JPG.format(), new File(imagePath));
                list.add(imagePath);
            }
            // 关闭文件, 不然该pdf文件会一直被占用
            pdf.close();
        } catch (IOException e) {
            throw new PdfOptException(e);
        }
        return list;
    }

    /**
     * PDF 文件转单张图片
     *
     * @param pdfPath pdf 源文件
     * @param descPath 转换后图片存储路径
     */
    public static void pdf2Image(String pdfPath, String descPath) {
        try {
            InputStream is = new FileInputStream(pdfPath);
            PDDocument pdf = PDDocument.load(is);
            int actSize  = pdf.getNumberOfPages();
            List<BufferedImage> images = new ArrayList<>();
            for (int i = 0; i < actSize; i++) {
                BufferedImage  image = new PDFRenderer(pdf).renderImageWithDPI(i,130, ImageType.RGB);
                images.add(image);
            }
            imagesMerge(images, descPath);
            is.close();
        } catch (IOException e) {
            throw new PdfOptException(e);
        }
    }

    /**
     * 纵向处理图片, 将宽度相同的图片，竖向追加在一起. <b>注意：宽度必须相同</b>
     *
     * @param images 图片集
     * @param descPath 结果路径
     */
    private static void imagesMerge(List<BufferedImage> images, String descPath) {
        if (images == null || images.size() <= 0) {
            System.out.println("图片数组为空!");
            return;
        }
        try {
            // 总高度, 总宽度, 临时的高度或保存偏移高度, 临时的高度(主要保存每个高度), 图片的数量
            int height = 0, width = 0, _height, __height, imageCnt = images.size();
            // 保存每个文件的高度
            int[] heightArray = new int[imageCnt];
            // 保存图片流
            BufferedImage buffer;
            // 保存所有的图片的RGB
            List<int[]> imgRGB = new ArrayList<>();
            // 保存一张图片中的RGB数据
            int[] _imgRGB;
            for (int i = 0; i < imageCnt; i++) {
                buffer = images.get(i);
                // 图片高度
                heightArray[i] = _height = buffer.getHeight();
                if (i == 0) {
                    // 图片宽度
                    width = buffer.getWidth();
                }
                // 获取总高度
                height += _height;
                // 从图片中读取RGB
                _imgRGB = new int[width * _height];
                _imgRGB = buffer.getRGB(0, 0, width, _height, _imgRGB, 0, width);
                imgRGB.add(_imgRGB);
            }
            _height = 0; // 设置偏移高度为0
            // 生成新图片
            BufferedImage imageResult = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
            for (int i = 0; i < imageCnt; i++) {
                __height = heightArray[i];
                // 计算偏移高度
                if (i != 0) {
                    _height += __height;
                }
                // 写入流中
                imageResult.setRGB(0, _height, width, __height, imgRGB.get(i), 0, width);
            }
            File descFile = new File(descPath);
            // 写图片
            ImageIO.write(imageResult, ImageFormatEnum.JPG.format(), descFile);
        } catch (Exception e) {
            throw new PdfOptException(e);
        }
    }




    /**
     * icepdf 方式pdf转图片
     *
     * @param pdfPath pdf文件路径
     * @param path    最终生成图片存储路径
     * @param scale   缩放比例
     * @param imageFormat   图片格式
     * @throws IOException io
     * @throws PDFException pfg
     * @throws PDFSecurityException pdfSecurity
     */
    public static void pdf2Img(String pdfPath, String path, float scale, ImageFormatEnum imageFormat)
            throws IOException, PDFException, PDFSecurityException {
        Document document = new Document();
        document.setFile(pdfPath);
        // 缩放比例
//        float scale = 3f;
        // 旋转角度
        float rotation = 0f;

        for (int i = 0; i < document.getNumberOfPages(); i++) {
            BufferedImage image = (BufferedImage)
                    document.getPageImage(i, GraphicsRenderingHints.SCREEN, Page.BOUNDARY_BLEEDBOX, rotation, scale);
            RenderedImage rendImage = image;
            try {
                String imgName = i + imageFormat.suffix();
                System.out.println(imgName);
                File file = new File(path + imgName);
                ImageIO.write(rendImage, imageFormat.format(), file);
            } catch (IOException e) {
                throw new PdfOptException(e);
            }
            image.flush();
        }
        document.dispose();
    }

}