package org.springframework.boot.dubbo.filter;

import com.alibaba.dubbo.rpc.*;
import org.springframework.boot.dubbo.bo.InterfaceInfo;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author   喝咖啡的囊地鼠
 *
 */
public abstract class AbstractStatisticsFilter implements Filter {


    public static final Map<InterfaceInfo, Map<String, AtomicLong>> INTERFACE_STATISTICS = new ConcurrentHashMap<>();

    private static final long ZERO = 0L;

    /**
     * 获取invoker
     * @param invoker
     * @param invocation
     * @return
     */
    public abstract Invoker<?> getInvoker(Invoker<?> invoker, Invocation invocation);

    /**
     * invoker 调用
     * @param invoker
     * @param invocation
     * @return
     * @throws RpcException
     */
    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        incrementAndGet(buildInterfaceInfo(getInvoker(invoker,invocation)), invocation.getMethodName());
        return invoker.invoke(invocation);
    }


    /**
     * 获取统计信息
     * @param info
     * @param methodName
     * @return
     */
    public static long get(InterfaceInfo info, String methodName) {
        if (!INTERFACE_STATISTICS.containsKey(info)){
            return ZERO;
        }
        AtomicLong ct = INTERFACE_STATISTICS.get(info).get(methodName);
        return ct == null ? ZERO : ct.get();
    }

    /**
     * 建立接口信息
     * @param invoker
     * @return
     */
    private InterfaceInfo buildInterfaceInfo(Invoker<?> invoker){
        return new InterfaceInfo(invoker);
    }

    /**
     * 计数
     * @param info
     * @param methodName
     */
    private synchronized static void incrementAndGet(InterfaceInfo info, String methodName) {
        if (!INTERFACE_STATISTICS.containsKey(info)){
            INTERFACE_STATISTICS.put(info,new ConcurrentHashMap<>(16));
        }
        AtomicLong ct = INTERFACE_STATISTICS.get(info).get(methodName);
        if (ct==null){
            ct = new AtomicLong(ZERO);
            INTERFACE_STATISTICS.get(info).put(methodName,ct);
        }
        ct.incrementAndGet();
    }
}
