package dec.service.common.config.feign.encoder;

import dec.service.common.utils.CommonUtils;
import feign.RequestTemplate;
import feign.codec.EncodeException;
import feign.codec.Encoder;
import feign.form.FormEncoder;
import feign.form.MultipartFormContentProcessor;
import feign.form.spring.SpringManyMultipartFilesWriter;
import feign.form.spring.SpringSingleMultipartFileWriter;
import lombok.val;
import org.springframework.web.multipart.MultipartFile;

import java.lang.reflect.Type;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import static feign.form.ContentType.MULTIPART;
import static java.util.Collections.singletonMap;

/**
 * @author dec
 */

public class FeignSpringFormEncoder extends FormEncoder {

    /**
     * Constructor with the default Feign's encoder as a delegate.
     */
    public FeignSpringFormEncoder () {
        this(new Default());
    }

    /**
     * Constructor with specified delegate encoder.
     *
     * @param delegate  delegate encoder, if this encoder couldn't encode object.
     */
    public FeignSpringFormEncoder (Encoder delegate) {
        super(delegate);
        val processor = (MultipartFormContentProcessor) getContentProcessor(MULTIPART);
        processor.addFirstWriter(new SpringSingleMultipartFileWriter());
        processor.addFirstWriter(new SpringManyMultipartFilesWriter());
    }

    @Override
    public void encode (Object object, Type bodyType, RequestTemplate template) throws EncodeException {
        if (bodyType.equals(MultipartFile[].class)) {
            val files = (MultipartFile[]) object;
            val data = new HashMap<String, Object>(files.length, 1.F);
            for (val file : files) {
                data.put(file.getName(), object);
            }
            super.encode(data, MAP_STRING_WILDCARD, template);
        } else if (bodyType.equals(MultipartFile.class)) {
            val file = (MultipartFile) object;
            val data = singletonMap(file.getName(), object);
            super.encode(data, MAP_STRING_WILDCARD, template);
        } else if (FeignSpringEncoder.isMultipartFileCollection(object)) {
            val collection = (Collection<?>) object;
            val data = new HashMap<String, Object>();
            if(CommonUtils.isNotEmpty(collection)) {
                MultipartFile[] files = new MultipartFile[collection.size()];
                files = collection.toArray(files);
                data.put(files[0].getName(), files);
            }

            super.encode(data, MAP_STRING_WILDCARD, template);
        } else if (FeignSpringEncoder.isMultipartFileMap(object)) {
            val map = (Map<?, ?>) object;
            val data = new HashMap<String, Object>();
            Iterator iterator = map.keySet().iterator();
            while (iterator.hasNext()) {
                val key = iterator.next();
                val value = map.get(key);
                if(value.getClass().equals(MultipartFile.class)) {
                    data.put(key.toString(), value);
                } else if(FeignSpringEncoder.isMultipartFileCollection(value)) {
                    val collection = (Collection<?>) value;
                    if(CommonUtils.isNotEmpty(value)) {
                        MultipartFile[] files = new MultipartFile[collection.size()];
                        files = collection.toArray(files);
                        data.put(files[0].getName(), files);
                    }
                }
            }

            super.encode(data, MAP_STRING_WILDCARD, template);
        } else {
            super.encode(object, bodyType, template);
        }
    }

}

