package dec.service.common.utils;

import dec.service.common.enums.ResponseCode;
import dec.service.common.vo.ResultVO;
import org.springframework.data.mongodb.core.query.Criteria;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.regex.Pattern;

/**
 * 通用工具类
 * @author dec
 */
public class CommonUtils {

    /**
     * common java.time.LocalDateTime formatter
     */
    private static DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    /**
     * common java.time.LocalDate formatter
     */
    private static DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    /**
     * 本机ip地址
     */
    private static InetAddress localhostIPAddress = null;

    /**
     * check object is null or not
     * @param object
     * @return
     */
    public static boolean isEmpty(Object object) {
        if(object == null) {
            return true;
        } else if(object instanceof Collection) {
            return ((Collection) object).isEmpty();
        } else if(object instanceof String) {
            return ((String) object).trim().isEmpty();
        } else {
            return false;
        }
    }

    /**
     * check object is null or not
     * @param object
     * @return
     */
    public static boolean isNotEmpty(Object object) {
        return !isEmpty(object);
    }

    /**
     * generate UUID string without '-'
     * @return
     */
    public static String UUID() {
        return UUID.randomUUID().toString().replaceAll("-", "");
    }

    /**
     * 判断相等
     * @param a
     * @param b
     * @return
     */
    public static boolean isEquals(Object a, Object b) {
        if(a != null && b != null) {
            return a.equals(b);
        }
        return false;
    }

    /**
     * 返回API调用结果统一封装对象: success
     * @param message
     * @return
     */
    public static ResultVO resultSuccess(String message) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setCode(ResponseCode.SERVICE_NORMAL.getCode());

        return result;
    }

    /**
     * 返回API调用结果统一封装对象: success
     * @param message
     * @param object
     * @return
     */
    public static ResultVO resultSuccess(String message, Object object) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setResult(object);
        result.setCode(ResponseCode.SERVICE_NORMAL.getCode());

        return result;
    }

    /**
     * 返回API调用结果统一封装对象: success
     * @param message
     * @param total
     * @param object
     * @return
     */
    public static ResultVO resultSuccess(String message, long total, Object object) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setTotal(total);
        result.setResult(object);
        result.setCode(ResponseCode.SERVICE_NORMAL.getCode());

        return result;
    }

    /**
     * 返回API调用结果统一封装对象: error
     * @param message
     * @return
     */
    public static ResultVO resultError(String message) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setCode(ResponseCode.SERVICE_INTERNAL_ERROR.getCode());

        return result;
    }

    /**
     * 返回API调用结果统一封装对象: error
     * @param message
     * @param code
     * @return
     */
    public static ResultVO resultError(String message, String code) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setCode(code);

        return result;
    }

    /**
     * 生成权限应用的版本
     * @param code
     * @return
     */
    public static String generateApplicationVersion(String code) {
        return code + "#" + UUID();
    }

    /**
     * LocalDateTime 格式化为字符串
     * @param dateTime
     * @return
     */
    public static String formatDateTime(LocalDateTime dateTime) {
        if(null == dateTime) {
            return "";
        }
        return dateTimeFormatter.format(dateTime);
    }

    /**
     * 格式化LocalDateTime字符串转换为LocalDateTime
     * @param dateTimeString
     * @return
     */
    public static LocalDateTime parseDateTime(String dateTimeString) {
        if(isEmpty(dateTimeString)) {
            return null;
        }
        return LocalDateTime.parse(dateTimeString, dateTimeFormatter);
    }

    /**
     * LocalDate 格式化为字符串
     * @param date
     * @return
     */
    public static String formatDate(LocalDate date) {
        if(null == date) {
            return "";
        }
        return dateFormatter.format(date);
    }

    /**
     * 格式化LocalDate字符串转换为LocalDate
     * @param dateString
     * @return
     */
    public static LocalDate parseDate(String dateString) {
        if(isEmpty(dateString)) {
            return null;
        }
        return LocalDate.parse(dateString, dateFormatter);
    }

    /**
     * 生成mongodb模糊搜索的正则条件
     * @param searchText
     * @return
     */
    public static Pattern generateSearchPattern(String searchText) {
        return Pattern.compile("^.*" + searchText + ".*$", Pattern.CASE_INSENSITIVE);
    }

    /**
     * 生成数据库查询条件： and
     * @param criteria
     * @return
     */
    public static Criteria generateAndCriteria(Criteria ...criteria) {
        List<Criteria> list = new ArrayList<>();
        for (Criteria criterion : criteria) {
            if(isNotEmpty(criterion)) {
                list.add(criterion);
            }
        }
        if(list.size() > 0) {
            Criteria[] arr = new Criteria[list.size()];
            return new Criteria().andOperator(list.toArray(arr));
        }
        return new Criteria();
    }

    /**
     * 生成数据库查询条件： or
     * @param criteria
     * @return
     */
    public static Criteria generateOrCriteria(Criteria ...criteria) {
        List<Criteria> list = new ArrayList<>();
        for (Criteria criterion : criteria) {
            if(isNotEmpty(criterion)) {
                list.add(criterion);
            }
        }
        if(list.size() > 0) {
            Criteria[] arr = new Criteria[list.size()];
            return new Criteria().orOperator(list.toArray(arr));
        }
        return new Criteria();
    }

    /**
     * 获取本机ip地址
     * @return
     */
    public static InetAddress getLocalHostAddress() throws IOException {
        if(isEmpty(localhostIPAddress)) {
            localhostIPAddress = initLocalHostAddress();
        }
        return localhostIPAddress;
    }

    private static InetAddress initLocalHostAddress() throws IOException {
        InetAddress candidateAddress = null;
        Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
        while (networkInterfaces.hasMoreElements()) {
            NetworkInterface iface = networkInterfaces.nextElement();
            for (Enumeration<InetAddress> inetAddrs = iface.getInetAddresses(); inetAddrs.hasMoreElements(); ) {
                InetAddress inetAddr = inetAddrs.nextElement();
                // 排除loopback回环类型地址
                if (!inetAddr.isLoopbackAddress()) {
                    if (inetAddr.isSiteLocalAddress()) {
                        // 如果是site-local地址， 直接返回
                        return inetAddr;
                    }

                    // 若不是site-local地址 那就记录下该地址当作候选
                    if (candidateAddress == null) {
                        candidateAddress = inetAddr;
                    }

                }
            }
        }

        if(isEmpty(candidateAddress)) {
            candidateAddress = InetAddress.getLocalHost();
        }
        return candidateAddress;
    }

}
