package dec.service.common.utils;

import dec.service.common.enums.ResponseCode;
import dec.service.common.vo.ResultVO;
import org.springframework.data.mongodb.core.query.Criteria;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.regex.Pattern;

/**
 * 通用工具类
 * @author dec
 */
public class CommonUtils {

    /**
     * common java.time.LocalDateTime formatter
     */
    private static DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    /**
     * common java.time.LocalDate formatter
     */
    private static DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    /**
     * check object is null or not
     * @param object
     * @return
     */
    public static boolean isEmpty(Object object) {
        if(object == null) {
            return true;
        } else if(object instanceof Collection) {
            return ((Collection) object).isEmpty();
        } else if(object instanceof String) {
            return ((String) object).trim().isEmpty();
        } else {
            return false;
        }
    }

    /**
     * check object is null or not
     * @param object
     * @return
     */
    public static boolean isNotEmpty(Object object) {
        return !isEmpty(object);
    }

    /**
     * generate UUID string without '-'
     * @return
     */
    public static String UUID() {
        return UUID.randomUUID().toString().replaceAll("-", "");
    }

    /**
     * 返回API调用结果统一封装对象: success
     * @param message
     * @return
     */
    public static ResultVO resultSuccess(String message) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setCode(ResponseCode.SERVICE_NORMAL.getCode());

        return result;
    }

    /**
     * 返回API调用结果统一封装对象: success
     * @param message
     * @param object
     * @return
     */
    public static ResultVO resultSuccess(String message, Object object) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setResult(object);
        result.setCode(ResponseCode.SERVICE_NORMAL.getCode());

        return result;
    }

    /**
     * 返回API调用结果统一封装对象: success
     * @param message
     * @param total
     * @param object
     * @return
     */
    public static ResultVO resultSuccess(String message, long total, Object object) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setTotal(total);
        result.setResult(object);
        result.setCode(ResponseCode.SERVICE_NORMAL.getCode());

        return result;
    }

    /**
     * 返回API调用结果统一封装对象: error
     * @param message
     * @return
     */
    public static ResultVO resultError(String message) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setCode(ResponseCode.SERVICE_INTERNAL_ERROR.getCode());

        return result;
    }

    /**
     * 返回API调用结果统一封装对象: error
     * @param message
     * @param code
     * @return
     */
    public static ResultVO resultError(String message, String code) {
        ResultVO result = new ResultVO();
        result.setTotal(0L);
        result.setMessage(message);
        result.setCode(code);

        return result;
    }

    /**
     * 生成权限应用的版本
     * @param code
     * @return
     */
    public static String generateApplicationVersion(String code) {
        return code + "#" + UUID();
    }

    /**
     * LocalDateTime 格式化为字符串
     * @param dateTime
     * @return
     */
    public static String formatDateTime(LocalDateTime dateTime) {
        if(null == dateTime) {
            return "";
        }
        return dateTimeFormatter.format(dateTime);
    }

    /**
     * LocalDate 格式化为字符串
     * @param date
     * @return
     */
    public static String formatDate(LocalDate date) {
        if(null == date) {
            return "";
        }
        return dateFormatter.format(date);
    }

    /**
     * 生成mongodb模糊搜索的正则条件
     * @param searchText
     * @return
     */
    public static Pattern generateSearchPattern(String searchText) {
        return Pattern.compile("^.*" + searchText + ".*$", Pattern.CASE_INSENSITIVE);
    }

    /**
     * 生成数据库查询条件： and
     * @param criteria
     * @return
     */
    public static Criteria generateAndCriteria(Criteria ...criteria) {
        List<Criteria> list = new ArrayList<>();
        for (Criteria criterion : criteria) {
            if(isNotEmpty(criterion)) {
                list.add(criterion);
            }
        }
        Criteria[] arr = new Criteria[list.size()];
        return new Criteria().andOperator(list.toArray(arr));
    }

    /**
     * 生成数据库查询条件： or
     * @param criteria
     * @return
     */
    public static Criteria generateOrCriteria(Criteria ...criteria) {
        List<Criteria> list = new ArrayList<>();
        for (Criteria criterion : criteria) {
            if(isNotEmpty(criterion)) {
                list.add(criterion);
            }
        }
        Criteria[] arr = new Criteria[list.size()];
        return new Criteria().orOperator(list.toArray(arr));
    }

}
