package io.ghostwriter.rt.tracer;

import io.ghostwriter.Tracer;
import io.ghostwriter.rt.tracer.serializer.StringSerializer;
import io.ghostwriter.rt.tracer.serializer.TracerSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Objects;

public class GhostWriterTracer implements Tracer {

    private static Logger LOG = LoggerFactory.getLogger(GhostWriterTracer.class);

    private final TracerSerializer serializer;

    private final Indentation indentation;

    public GhostWriterTracer(TracerSerializer serializer) {
        this.serializer = Objects.requireNonNull(serializer);
        this.indentation = new Indentation();
    }

    public GhostWriterTracer() {
        this(new StringSerializer());
    }

    @Override
    public void entering(Object source, String method, Object... params) {
        final int DEFAULT_CAPACITY = 255;
        StringBuffer sb = new StringBuffer(DEFAULT_CAPACITY);

        final String context = serializer.serialize(source);

        indentation.apply(sb).append(context).append(".").append(method).append("(");
        appendParameters(sb, params).append(") {");

        write(sb);

        indentation.indent();
    }

    @Override
    public void exiting(Object source, String method, Object returnValue) {
        final int DEFAULT_CAPACITY = 64;
        final StringBuffer sb = new StringBuffer(DEFAULT_CAPACITY);

        indentation.dedent();
        final String value = serializer.serialize(returnValue);
        indentation.apply(sb).append("} = ").append(value);

        write(sb);
    }

    @Override
    public void exiting(Object source, String method) {
        final int DEFAULT_CAPACITY = 64;
        final StringBuffer sb = new StringBuffer(DEFAULT_CAPACITY);

        indentation.dedent();
        indentation.apply(sb).append("}");

        write(sb);
    }

    @Override
    public void valueChange(Object source, String method, String variable, Object value) {
        final int DEFAULT_CAPACITY = 64;
        final StringBuffer sb = new StringBuffer(DEFAULT_CAPACITY);

        final String strValue = serializer.serialize(value);
        indentation.apply(sb).append(variable).append(" = ").append(strValue);

        write(sb);
    }

    @Override
    public void onError(Object source, String method, Throwable error) {
        final int DEFAULT_CAPACITY = 128;
        final StringBuffer sb = new StringBuffer(DEFAULT_CAPACITY);

        // NOTE(snorbi07): this might not be the 'best' way to represent an error ...
        final String errorStr = serializer.serialize(error);
        indentation.apply(sb).append("<Exception> = ").append(errorStr);

        write(sb);
    }

    private StringBuffer appendParameters(StringBuffer sb, Object[] params) {
        final int NUMBER_OF_ENTRIES_PER_PARAMETERS = 2;

        for (int i = 0; i < params.length; i += NUMBER_OF_ENTRIES_PER_PARAMETERS) {
            Object parameterName = params[i];
            Object parameterValue = params[i + 1];
            String name = serializer.serialize(parameterName);
            String value = serializer.serialize(parameterValue);
            sb.append(name).append(" = ").append(value);

            boolean isFinalParameter = i == (params.length - NUMBER_OF_ENTRIES_PER_PARAMETERS);
            if (!isFinalParameter) {
                sb.append(", ");
            }
        }

        return sb;
    }

    // FIXME(snorbi07): this should not be strictly coupled to an implementation! Think of testing as well.
    private void write(StringBuffer sb) {
        LOG.trace(sb.toString());
    }

}
